package eventbus

import (
	"context"
	"fmt"

	achievement "code.justin.tv/cb/achievements/internal/achievement/nfollowers"
	"code.justin.tv/cb/achievements/internal/clients/sqs"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	eventbus "code.justin.tv/eventbus/client"
	userfollow "code.justin.tv/eventbus/schema/pkg/user_follow_user"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type userFollowUserHandler struct {
	workerSQSClient sqs.SQS
	sqsstatsd       stats.StatsdClient
}

type UserFollowUserHandler interface {
	ProcessNFollowersUpdate(ctx context.Context, _ *eventbus.Header, event *userfollow.Create) error
}

func NewUserFollowUserHandler(workerSQSClient sqs.SQS, statsdClient stats.StatsdClient) UserFollowUserHandler {
	return &userFollowUserHandler{
		workerSQSClient: workerSQSClient,
		sqsstatsd:       statsdClient,
	}
}

// processNFollowersUpdate listens to the eventbus UserFollowUserCreate topic
// then sends the events to the worker via SNS
func (h *userFollowUserHandler) ProcessNFollowersUpdate(ctx context.Context, _ *eventbus.Header, event *userfollow.Create) error {
	err := h.workerSQSClient.Send(ctx, achievement.Key, &achievement.Input{
		Channels: []*achievement.ChannelFollowers{
			{
				ChannelID:   event.ToUserId,
				FollowCount: int(event.ToUserFollowerCount),
			},
		},
	})

	go h.saveSQSState(achievement.Key, 1)

	if err != nil {
		return errors.Wrap(err, "error: failed to send n_followers event to worker sqs from eventbus listener")
	}

	return nil
}

// SaveSQSState saves message counts to statsd whenever we create a batch of events from eventbus
// so we can monitor the state of the queue
func (h *userFollowUserHandler) saveSQSState(achievement string, size int64) {
	err := h.sqsstatsd.Inc(fmt.Sprintf("sqs.%s.messages", achievement), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = h.sqsstatsd.Inc(fmt.Sprintf("sqs.%s.messages", achievement), int64(size)-1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = h.sqsstatsd.Inc(fmt.Sprintf("sqs.%s.sent", achievement), int64(size), 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}
