package moneypenny

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"math/rand"
	"net/http"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/proxy"
)

const backoffCeiling = 30

// HTTPClient is a wrapper for http.Client that also contains a host
// 	for prod/dev splits
type HTTPClient struct {
	host   string
	client *http.Client
}

// NewHTTPClient instantiates and returns an HTTPClient
func NewHTTPClient(host string) (*HTTPClient, error) {
	var dialer proxy.Dialer
	dialer = proxy.Direct // default to direct

	// The reinvite_failed_affiliates script is run from a developer laptop and
	// requires an HTTP SOCKS proxy to connect to moneypenny.
	httpProxy, httpProxyPresent := os.LookupEnv("HTTP_PROXY")
	if httpProxyPresent {
		httpProxyURL, err := url.Parse(httpProxy)
		if err != nil {
			return nil, fmt.Errorf("invalid proxy url %q: %w", httpProxyURL, err)
		}

		dialer, err = proxy.FromURL(httpProxyURL, proxy.Direct)
		if err != nil {
			return nil, fmt.Errorf("proxy from url: %w", err)
		}
	}
	httpTransport := &http.Transport{Dial: dialer.Dial}

	return &HTTPClient{
		host: host,
		client: &http.Client{
			Timeout:   10 * time.Second,
			Transport: httpTransport,
		},
	}, nil
}

const ldap = "cb-achievements"

// PostAffiliateInvitation invites a channel to become an affiliate!
func (h *HTTPClient) PostAffiliateInvitation(ctx context.Context, id string) (*Invitation, error) {
	log.WithFields(log.Fields{
		"channel_id": id,
		"timestamp":  time.Now(),
	}).Info("moneypenny: sending affiliate invitation")

	url := fmt.Sprintf("%s/payout/invite/%s", h.host, id)
	buffer := &bytes.Buffer{}

	body := InvitationRequestBody{
		Category: "affiliate",
		Tags:     []string{},
		Features: &InvitationFeatures{
			Bits:        true,
			Subs:        true,
			PreRollAds:  true,
			PostRollAds: true,
		},
	}

	err := json.NewEncoder(buffer).Encode(&body)
	if err != nil {
		return nil, err
	}

	req, err := http.NewRequest(http.MethodPost, url, buffer)
	if err != nil {
		return nil, err
	}

	req = req.WithContext(ctx)
	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Twitch-LDAP-Login", ldap)

	resp, err := h.client.Do(req)
	if err != nil {
		return nil, errors.Wrap(err, "moneypenny: failed to make HTTP request")
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			log.WithError(err).Error("moneypenny: failed to close response body")
		}
	}()

	if resp.StatusCode != http.StatusOK {
		body := Error{}

		err = json.NewDecoder(resp.Body).Decode(&body)
		if err != nil {
			return nil, errors.Wrap(err, "moneypenny: failed to decode non-200 response body to JSON")
		}

		switch {
		case strings.Contains(strings.ToLower(body.Message), "not allowed to downgrade invite"):
			return nil, ErrCannotDowngradeInvite
		case strings.Contains(strings.ToLower(body.Message), "cannot re-invite channel"):
			return nil, ErrCannotReinvite
		case strings.Contains(strings.ToLower(body.Message), "has incomplete workflow state"):
			return nil, ErrIncompleteWorkflow
		default:
			return nil, errors.Errorf("moneypenny: unexpected status code (%d) and message: %s", body.Status, body.Message)
		}
	}

	output := &Invitation{}

	err = json.NewDecoder(resp.Body).Decode(output)
	if err != nil {
		return nil, errors.Wrap(err, "moneypenny: failed to decode response body to JSON")
	}

	return output, nil
}

// RandomWait sleeps for a maximum of maxSeconds * 10. This is used to slow down requests to
// moneypenny because they can't handle our volume.
func (h *HTTPClient) RandomWait(maxSeconds int) {
	if maxSeconds > backoffCeiling {
		maxSeconds = backoffCeiling
	}

	backoff := rand.Intn(maxSeconds)
	time.Sleep(time.Duration(backoff) * time.Second)
}
