package moneypenny

import (
	"context"
	"errors"
)

// Expected errors from Moneypenny:
var (
	ErrCannotDowngradeInvite = errors.New("moneypenny: cannot downgrade invite")
	ErrCannotReinvite        = errors.New("moneypenny: cannot reinvite")
	ErrIncompleteWorkflow    = errors.New("moneypenny: incomplete workflow state")

	ErrInvitationNotFound = errors.New("moneypenny: cannot find invitation")
)

// Moneypenny is a generic moneypenny service interface, to be implemented
// by a specific package (or mock).
type Moneypenny interface {
	// PostAffiliateInvitation invites channel id to become a twitch affiliate
	PostAffiliateInvitation(ctx context.Context, id string) (*Invitation, error)

	// RandomWait does nothing but sleep for a random amount of time between 0 and the maxSeconds provided
	RandomWait(maxSeconds int)
}

// UserPayoutStatus is a representation of a channelid's payout status
type UserPayoutStatus struct {
	ChannelID   string `json:"channel_id"`
	IsAffiliate bool   `json:"is_affiliate"`
	IsDeveloper bool   `json:"is_developer"`
	IsPartner   bool   `json:"is_partner"`
}

// Invitation is the successfully created invitation
type Invitation struct {
	ChannelID    string `json:"channel_id"`
	InvitationID string `json:"invitation_id"`
	Success      bool   `json:"success"`
}

// Error is how moneypenny communicates errors
type Error struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
}

// InvitationRequestBody is the body we will send to moneypenny to invite
// a channel
type InvitationRequestBody struct {
	Category string              `json:"category"`
	Tags     []string            `json:"tags"`
	Features *InvitationFeatures `json:"features"`
}

// InvitationFeatures are the features we want to enable in a moneypenny invitiation
// request
type InvitationFeatures struct {
	Bits        bool `json:"bits"`
	Subs        bool `json:"subs"`
	PreRollAds  bool `json:"pre_roll_ads"`
	PostRollAds bool `json:"post_roll_ads"`
}
