package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// BroadcastNHours30Days queries Redshift to see which channels have broadcast
// more than an arbitrary number of minutes over the last 30 days. The minutesThreshold
// parameter filters for channels with a minimum level of minutes broadcast.
//
// It is recommended that this query run every hour to avoid time zone/day
// rollover issues. Only returns results for channels that have broadcast in the last 30 days.
func (c *Client) BroadcastNHours30Days(ctx context.Context, minutesThreshold int) ([]*MinutesBroadcastAggregate, error) {
	statement := `
        SELECT channel_id,
               count(channel_id) AS minute_count
          FROM spade.minute_broadcast
         WHERE date >= $1 AND date < $2
           AND channel_id IS NOT NULL
           AND time_utc > GETDATE() - interval '30 day'
           AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
           AND (game <> 'watch parties' OR game IS NULL)
      GROUP BY channel_id
        HAVING count(channel_id) >= $3
      ORDER BY minute_count DESC
    `
	var resultsAggregate []*MinutesBroadcastAggregate
	currTime := time.Now()
	thirtyDaysAgo := currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat)
	queryTimeRangeEnd := currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat)

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement, thirtyDaysAgo, queryTimeRangeEnd, minutesThreshold)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.broadcast_n_hours_30_days.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*MinutesBroadcastAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "pq: failed to query minutes_broadcast")
		}
	}

	c.stats.ExecutionTime("redshift.broadcast_n_hours_30_days.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":        statement,
				"minutesThreshold": minutesThreshold,
			}).Error("pq: failed to close rows")
		}
	}()

	for rows.Next() {
		channelAggregate := &MinutesBroadcastAggregate{}
		err = rows.Scan(
			&channelAggregate.ChannelID,
			&channelAggregate.Minutes,
		)

		resultsAggregate = append(resultsAggregate, channelAggregate)
	}

	err = rows.Err()
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"statement":        statement,
			"minutesThreshold": minutesThreshold,
		}).Error("pq: failed to scan rows")

		return nil, err
	}

	return resultsAggregate, nil
}
