package redshift

import (
	"context"
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type broadcastNHours30Days struct {
	suite.Suite
	redshiftClient *Client
	mock           sqlmock.Sqlmock

	thirtyDaysAgo             string
	queryTimeRangeEnd         string
	broadcastMinutesThreshold int
}

func TestBroadcastNHours30Days(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)
	currTime := time.Now()

	s := &broadcastNHours30Days{
		redshiftClient:            &Client{stub, stats.NewNoopClient()},
		mock:                      mock,
		thirtyDaysAgo:             currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat),
		queryTimeRangeEnd:         currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat),
		broadcastMinutesThreshold: 480,
	}

	suite.Run(t, s)
}

func (s *broadcastNHours30Days) TestNoRows_ReturnsEmptyArray() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.broadcastMinutesThreshold).
		WillReturnError(sql.ErrNoRows)

	results, _ := s.redshiftClient.BroadcastNHours30Days(context.Background(), s.broadcastMinutesThreshold)
	s.Equal(results, []*MinutesBroadcastAggregate{})

	err := s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *broadcastNHours30Days) TestUnexpectedError() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.broadcastMinutesThreshold).
		WillReturnError(sql.ErrTxDone)

	results, err := s.redshiftClient.BroadcastNHours30Days(context.Background(), s.broadcastMinutesThreshold)
	s.Error(err)
	s.Nil(results)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *broadcastNHours30Days) TestSuccess() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.broadcastMinutesThreshold).
		WillReturnRows(
			sqlmock.NewRows([]string{"channel_id", "minute_count"}).
				AddRow("123456", "500"),
		)

	results, err := s.redshiftClient.BroadcastNHours30Days(context.Background(), s.broadcastMinutesThreshold)
	s.NoError(err)

	s.Equal(1, len(results))

	if len(results) > 0 {
		s.Equal("123456", results[0].ChannelID)
		s.Equal(500, results[0].Minutes)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}
