package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NBroadcastDays30Days queries Redshift to count the unique days of week that a channel has broadcast.
func (c *Client) NBroadcastDays30Days(ctx context.Context, daysThreshold int) ([]*UniqueDaysBroadcastAggregate, error) {
	statement := `
        SELECT channel_id,
               COUNT(DISTINCT date_part('day', time_utc)) AS unique_days
          FROM spade.minute_broadcast
         WHERE date >= $1 AND date < $2
           AND channel_id IS NOT NULL
           AND time_utc > GETDATE() - interval '30 day'
           AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
           AND (game <> 'watch parties' OR game IS NULL)
      GROUP BY channel_id
        HAVING COUNT(DISTINCT date_part('day', time_utc)) >= $3
      ORDER BY unique_days DESC
    `
	var results []*UniqueDaysBroadcastAggregate
	currTime := time.Now()
	thirtyDaysAgo := currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat)
	queryTimeRangeEnd := currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat)

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement, thirtyDaysAgo, queryTimeRangeEnd, daysThreshold)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_broadcast_days_30_days.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*UniqueDaysBroadcastAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "pq: failed to query broadcast days")
		}
	}

	c.stats.ExecutionTime("redshift.n_broadcast_days_30_days.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":     statement,
				"daysThreshold": daysThreshold,
			}).Error("pq: failed to close rows")
		}
	}()

	for rows.Next() {
		aggregate := &UniqueDaysBroadcastAggregate{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.UniqueDays,
		)

		results = append(results, aggregate)
	}

	err = rows.Err()
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"statement":     statement,
			"daysThreshold": daysThreshold,
		}).Error("pq: failed to scan rows")

		return nil, err
	}

	return results, nil
}
