package redshift

import (
	"context"
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type nBroadcastDays30Days struct {
	suite.Suite
	redshiftClient *Client
	mock           sqlmock.Sqlmock

	thirtyDaysAgo          string
	queryTimeRangeEnd      string
	broadcastDaysThreshold int
}

func TestNBroadcastDays30Days(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)
	currTime := time.Now()

	s := &nBroadcastDays30Days{
		redshiftClient:         &Client{stub, stats.NewNoopClient()},
		mock:                   mock,
		thirtyDaysAgo:          currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat),
		queryTimeRangeEnd:      currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat),
		broadcastDaysThreshold: 7,
	}

	suite.Run(t, s)
}

func (s *nBroadcastDays30Days) TestNoRows_ReturnsEmptyArray() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.broadcastDaysThreshold).
		WillReturnError(sql.ErrNoRows)

	results, _ := s.redshiftClient.NBroadcastDays30Days(context.Background(), s.broadcastDaysThreshold)
	s.Equal(results, []*UniqueDaysBroadcastAggregate{})

	err := s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *nBroadcastDays30Days) TestUnexpectedError() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.broadcastDaysThreshold).
		WillReturnError(sql.ErrTxDone)

	results, err := s.redshiftClient.NBroadcastDays30Days(context.Background(), s.broadcastDaysThreshold)
	s.Error(err)
	s.Nil(results)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *nBroadcastDays30Days) TestSuccess() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.broadcastDaysThreshold).
		WillReturnRows(
			sqlmock.NewRows([]string{"channel_id", "unique_days"}).
				AddRow("123456", "10"),
		)

	results, err := s.redshiftClient.NBroadcastDays30Days(context.Background(), s.broadcastDaysThreshold)
	s.NoError(err)

	s.Equal(1, len(results))

	if len(results) > 0 {
		s.Equal("123456", results[0].ChannelID)
		s.Equal(10, results[0].UniqueDays)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}
