package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NConcurrents30Days queries Redshift to retrieve the average ccus
// a channel had over the last 30 days. It is recommended that this query run every hour
// to avoid time zone/day rollover issues.
// This query only returns results for channels that have broadcast in the last 2 hours.
func (c *Client) NConcurrents30Days(ctx context.Context, avgCCUThreshold float64) ([]*ConcurrentViewersBroadcastAverage, error) {
	statement := `
        WITH average_ccu_in_window AS
        (
            SELECT mb.channel_id,
                   AVG(cc.total::float) AS average_ccu,
                   COUNT(mb.channel_id) AS minutes_broadcast
              FROM spade.minute_broadcast mb
         LEFT JOIN spade.channel_concurrents cc
                ON mb.channel_id = cc.channel_id AND date_trunc('minute', mb.time_utc) = date_trunc('minute', cc.time_utc) AND cc.date >= $1 AND cc.date < $2
             WHERE mb.date >= $1 AND mb.date < $2
               AND mb.channel_id IS NOT NULL
               AND mb.time_utc > GETDATE() - interval '30 day'
			   AND (mb.broadcaster_software IS NULL OR (mb.broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
			   AND cc.content_mode <> 'prime_video_watch_party'
               AND (mb.game <> 'watch parties' OR mb.game IS NULL)
          GROUP BY mb.channel_id
        )

        SELECT channel_id,
               average_ccu
          FROM average_ccu_in_window
         WHERE minutes_broadcast > 60
           AND average_ccu >= $3
      ORDER BY average_ccu DESC
    `
	var results []*ConcurrentViewersBroadcastAverage
	currTime := time.Now()
	thirtyDaysAgo := currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat)
	queryTimeRangeEnd := currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat)

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement, thirtyDaysAgo, queryTimeRangeEnd, avgCCUThreshold)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_concurrent_30_days.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*ConcurrentViewersBroadcastAverage{}, nil
		default:
			return nil, errors.Wrap(err, "pq: failed to query for concurrent viewers")
		}
	}

	c.stats.ExecutionTime("redshift.n_concurrent_30_days.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":       statement,
				"avgCCUThreshold": avgCCUThreshold,
			}).Error("pq: failed to close rows for average ccu")
		}
	}()

	for rows.Next() {
		aggregate := &ConcurrentViewersBroadcastAverage{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.AverageCCU,
		)

		results = append(results, aggregate)
	}

	err = rows.Err()
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"statement":       statement,
			"avgCCUThreshold": avgCCUThreshold,
		}).Error("pq: failed to scan rows for average ccu")

		return nil, err
	}

	return results, nil
}
