package redshift

import (
	"context"
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type nConcurrents30Days struct {
	suite.Suite
	redshiftClient *Client
	mock           sqlmock.Sqlmock

	thirtyDaysAgo     string
	queryTimeRangeEnd string
	avgCCUThreshold   float64
}

func TestNConcurrents30Days(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)
	currTime := time.Now()

	s := &nConcurrents30Days{
		redshiftClient:    &Client{stub, stats.NewNoopClient()},
		mock:              mock,
		thirtyDaysAgo:     currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat),
		queryTimeRangeEnd: currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat),
		avgCCUThreshold:   3.0,
	}

	suite.Run(t, s)
}

func (s *nConcurrents30Days) TestNoRows_ReturnsEmptyArray() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.avgCCUThreshold).
		WillReturnError(sql.ErrNoRows)

	results, _ := s.redshiftClient.NConcurrents30Days(context.Background(), s.avgCCUThreshold)
	s.Equal(results, []*ConcurrentViewersBroadcastAverage{})

	err := s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *nConcurrents30Days) TestUnexpectedError() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.avgCCUThreshold).
		WillReturnError(sql.ErrTxDone)

	results, err := s.redshiftClient.NConcurrents30Days(context.Background(), s.avgCCUThreshold)
	s.Error(err)
	s.Nil(results)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *nConcurrents30Days) TestSuccess() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.thirtyDaysAgo, s.queryTimeRangeEnd, s.avgCCUThreshold).
		WillReturnRows(
			sqlmock.NewRows([]string{"channel_id", "average_ccu"}).
				AddRow("123456", "3.0"),
		)

	results, err := s.redshiftClient.NConcurrents30Days(context.Background(), s.avgCCUThreshold)
	s.NoError(err)

	s.Equal(1, len(results))

	if len(results) > 0 {
		s.Equal("123456", results[0].ChannelID)
		s.Equal(3.0, results[0].AverageCCU)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}
