package redshift

import (
	"context"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	createTemp = `
    CREATE TEMPORARY TABLE n_days_no_viewers_sessions(
        chunk integer encode delta,
        chunk_end_time timestamp encode delta,
        chunk_start_time timestamp,
        channel_id integer )
    distkey (channel_id)
    sortkey (channel_id, chunk_start_time);
    `
	populateTemp = `
    INSERT INTO n_days_no_viewers_sessions
    WITH with_gaps AS
    (
             SELECT   channel_id ,
                      (Date_trunc('minute', time_utc) - Lag(Date_trunc('minute', time_utc), 1) OVER (partition BY channel_id ORDER BY time_utc ASC)) AS gap ,
                      Date_trunc('minute', time_utc)                                                                                                 AS minute_trunc ,
                      time_utc
             FROM     logs.minute_broadcast
             WHERE    time_utc > Getdate() - interval '6 months'
             AND (game <> 'watch parties' OR game IS NULL)
             AND      (broadcaster_software IS NULL OR broadcaster_software <> 'watch_party')) , with_sessions AS
    (
             SELECT   sum(
                      CASE
                               WHEN gap > interval '15 minutes' THEN 1
                               ELSE 0
                      END) OVER (partition BY channel_id ORDER BY time_utc ASC rows BETWEEN UNBOUNDED PRECEDING AND CURRENT row) AS session ,
                      channel_id ,
                      time_utc
             FROM     with_gaps ) , complete_sessions AS
    (
             SELECT   min(time_utc) AS start_time ,
                      max(time_utc) AS end_time ,
                      CASE
                               WHEN max(time_utc) - min(time_utc) <= interval '48 hours' THEN 1
                               ELSE ceil((date_part('epoch', max(time_utc)) - date_part('epoch', min(time_utc))) / 60 / 60 / 24)
                      END AS chunks ,
                      channel_id
             FROM     with_sessions
             GROUP BY channel_id,
                      session ) , calculated_sessions AS
    (
              SELECT    integers.INDEX                                        AS chunk ,
                        end_time - (integers.INDEX - 1) * interval '24 hours' AS chunk_end_time ,
                        CASE
                                  WHEN chunks = 1 THEN start_time
                                  WHEN start_time > end_time - integers.INDEX * interval '24 hours' THEN start_time
                                  ELSE end_time              - integers.INDEX * interval '24 hours'
                        END AS chunk_start_time ,
                        channel_id
              FROM      complete_sessions
              LEFT JOIN logs.integer integers
              ON        complete_sessions.chunks >= integers.INDEX )
    SELECT * FROM calculated_sessions;
    `
	query = `
    WITH concurrents AS
    (
             SELECT   channel_id ,
                      time_utc ,
                      total
             FROM     logs.channel_concurrents
             WHERE    time_utc > Getdate() - interval '6 months'
                      AND logs.channel_concurrents.content_mode <> 'prime_video_watch_party'), consecutive_averages AS
    (
              SELECT    sessions.channel_id ,
                        avg(concurrents.total::float) AS average_ccu ,
                        sessions.chunk_start_time,
                        sum(
                        CASE
                                  WHEN avg(concurrents.total::float) < 3 THEN 1
                                  ELSE 0
                        END) OVER (partition BY sessions.channel_id ORDER BY sessions.chunk_start_time ASC rows BETWEEN UNBOUNDED PRECEDING AND CURRENT row) AS threshold ,
                        row_number() OVER (partition BY sessions.channel_id ORDER BY sessions.chunk_start_time)                                                    AS mod
              FROM      n_days_no_viewers_sessions AS sessions
              LEFT JOIN concurrents
              ON        concurrents.channel_id = sessions.channel_id
              AND       concurrents.time_utc >= sessions.chunk_start_time
              AND       concurrents.time_utc < sessions.chunk_end_time
              GROUP BY  sessions.channel_id,
                        sessions.chunk_start_time ), streaks AS
    (
             SELECT   count(*)      AS streak,
                      mod-threshold AS hash,
                      channel_id
             FROM     consecutive_averages
             WHERE    average_ccu < 3
             GROUP BY channel_id,
                      mod-threshold )
    SELECT   max(streak) AS count,
             channel_id
    FROM     streaks
    GROUP BY channel_id;
    `
	cleanup = `
    DROP TABLE n_days_no_viewers_sessions;
    `
)

// NDaysNoViewers queries Redshift to retrieve the number of consecutive sessions
// where a user averaged less than 3 ccus. It is recommended that this query run every hour
// to avoid time zone/day rollover issues.
// This query only returns results for channels that have broadcast in the last 2 hours.
func (c *Client) NDaysNoViewers(ctx context.Context) ([]*ConsecutiveSessionsNoViewersAggregate, error) {
	var results []*ConsecutiveSessionsNoViewersAggregate
	txn, err := c.db.BeginTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "redshift: failed to start tranasction to find sessions with no viewers by channelID")
	}

	defer func() {
		if err != nil {
			err = txn.Rollback()

		}
		err = txn.Commit()
	}()

	_, err = txn.ExecContext(ctx, createTemp)
	if err != nil {
		return nil, err
	}

	_, err = txn.ExecContext(ctx, populateTemp)
	if err != nil {
		return nil, err
	}

	rows, err := txn.QueryContext(ctx, query)
	if err != nil {
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift: failed to close rows when querying sessions with no viewers by channelID"

			log.WithError(err).Error(msg)
		}
	}()

	for rows.Next() {
		aggregate := &ConsecutiveSessionsNoViewersAggregate{}
		err = rows.Scan(
			&aggregate.ConsecutiveSessions,
			&aggregate.ChannelID,
		)

		results = append(results, aggregate)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift: error scanning rows when querying sessions with no viewers by channelID"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	_, err = txn.ExecContext(ctx, cleanup)
	if err != nil {
		return nil, err
	}

	return results, nil
}
