package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NFollowers queries Redshift for the last known follower count in the last
// 6 mo for each channel. This is meant to refresh checkmarks for users that
// not have had a follower recently
func (c *Client) NFollowers(ctx context.Context, followerThreshold int) ([]*NFollowersAggregate, error) {
	statement := `
        WITH recent_follows AS
        (
            SELECT target_user_id AS channel_id,
                   follow_count,
                   row_number() over (partition by target_user_id order by time_utc desc) as rank
              FROM spade.server_follow
             WHERE date >= $1 AND date < $2
               AND channel_id IS NOT NULL
               AND follow_count IS NOT NULL
        )

        SELECT channel_id,
               follow_count
          FROM recent_follows
         WHERE rank = 1
           AND follow_count >= $3
      ORDER BY channel_id DESC
    `
	var results []*NFollowersAggregate
	currTime := time.Now()
	fourDaysAgo := currTime.Truncate(24*time.Hour).AddDate(0, 0, -4).Format(sqlTimeFormat)
	queryTimeRangeEnd := currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat)

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement, fourDaysAgo, queryTimeRangeEnd, followerThreshold)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_followers.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*NFollowersAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "pq: failed to query for followers")
		}
	}

	c.stats.ExecutionTime("redshift.n_followers.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":         statement,
				"followerThreshold": followerThreshold,
			}).Error("pq: failed to close rows for followers")
		}
	}()

	for rows.Next() {
		aggregate := &NFollowersAggregate{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.Followers,
		)

		if aggregate.ChannelID != "" {
			results = append(results, aggregate)
		}
	}

	err = rows.Err()
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"statement":         statement,
			"followerThreshold": followerThreshold,
		}).Error("pq: failed to scan rows for followers")

		return nil, err
	}

	return results, nil
}
