package redshift

import (
	"context"
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type nFollowers struct {
	suite.Suite
	redshiftClient *Client
	mock           sqlmock.Sqlmock

	fourDaysAgo       string
	queryTimeRangeEnd string
	followerThreshold int
}

func TestNFollowers(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)
	currTime := time.Now()

	s := &nFollowers{
		redshiftClient:    &Client{stub, stats.NewNoopClient()},
		mock:              mock,
		fourDaysAgo:       currTime.Truncate(24*time.Hour).AddDate(0, 0, -4).Format(sqlTimeFormat),
		queryTimeRangeEnd: currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat),
		followerThreshold: 50,
	}

	suite.Run(t, s)
}

func (s *nFollowers) TestNoRows_ReturnsEmptyArray() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.fourDaysAgo, s.queryTimeRangeEnd, s.followerThreshold).
		WillReturnError(sql.ErrNoRows)

	results, _ := s.redshiftClient.NFollowers(context.Background(), s.followerThreshold)
	s.Equal(results, []*NFollowersAggregate{})

	err := s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *nFollowers) TestUnexpectedError() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.fourDaysAgo, s.queryTimeRangeEnd, s.followerThreshold).
		WillReturnError(sql.ErrTxDone)

	results, err := s.redshiftClient.NFollowers(context.Background(), s.followerThreshold)
	s.Error(err)
	s.Nil(results)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *nFollowers) TestSuccess() {
	s.mock.ExpectQuery("SELECT channel_id").
		WithArgs(s.fourDaysAgo, s.queryTimeRangeEnd, s.followerThreshold).
		WillReturnRows(
			sqlmock.NewRows([]string{"channel_id", "follower_count"}).
				AddRow("123456", "75"),
		)

	results, err := s.redshiftClient.NFollowers(context.Background(), s.followerThreshold)
	s.NoError(err)

	s.Equal(1, len(results))

	if len(results) > 0 {
		s.Equal("123456", results[0].ChannelID)
		s.Equal(75, results[0].Followers)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}
