package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NHoursLifetime queries Redshift to count the number of minutes that
// that a channel has broadcast. We will run this query hourly (dow will be defined
// from the perspective of twitch server time, but we will run this hourly to get achievements
// to users asap).
// This query only returns results for channels that have broadcast in the last 2 hours.
//
// This query does not run on a tahoe replica: The need for querying lifetime data is not well supported
// in a tahoe replica, and therefore we continue to use the achievements redshift cluster.
func (c *Client) NHoursLifetime(ctx context.Context) ([]*LifeTimeBroadcastDurationAggregate, error) {
	statement := `
        WITH total_time_broadcast AS
        (
            SELECT channel_id,
                   COUNT(channel_id) AS minute_count
              FROM logs.minute_broadcast
             WHERE channel_id IS NOT NULL
             AND (game <> 'watch parties' OR game IS NULL)
          GROUP BY channel_id
        ),

        recently_broadcast AS
        (
            SELECT DISTINCT channel_id
              FROM logs.minute_broadcast
             WHERE channel_id IS NOT NULL
               AND time_utc > GETDATE() - interval '2 hour'
               AND (game <> 'watch parties' OR game IS NULL)
			   AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
        )

        SELECT recently_broadcast.channel_id,
               total_time_broadcast.minute_count
          FROM recently_broadcast
     LEFT JOIN total_time_broadcast
            ON recently_broadcast.channel_id = total_time_broadcast.channel_id
    `
	var results []*LifeTimeBroadcastDurationAggregate

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_hours_lifetime.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*LifeTimeBroadcastDurationAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "pq: failed to query hours lifetime")
		}
	}

	c.stats.ExecutionTime("redshift.n_hours_lifetime.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift: failed to close rows when selecting lifetime broadcast stats by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		aggregate := &LifeTimeBroadcastDurationAggregate{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.MinutesBroadcast,
		)

		results = append(results, aggregate)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift: error scanning rows when selecting lifetime viewing stats by channelID"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
