package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NMinuteWatchedLifetime queries Redshift to count the number of minutes watched
// for a channel. We will run this query hourly (dow will be defined
// from the perspective of twitch server time, but we will run this hourly to get achievements
// to users asap).
// This query only returns results for channels that have broadcast in the last 2 hours.
//
// This query does not run on a tahoe replica: The need for querying lifetime data is not well supported
// in a tahoe replica, and therefore we continue to use the achievements redshift cluster.
func (c *Client) NMinuteWatchedLifetime(ctx context.Context) ([]*LifeTimeViewingDurationAggregate, error) {
	statement := `
        WITH total_time_watched AS
        (
            SELECT channel_id,
                   SUM(total) AS total_minutes_watched
              FROM logs.channel_concurrents
             WHERE channel_id IS NOT NULL
          GROUP BY channel_id
        ),

        recently_broadcast AS
        (
            SELECT DISTINCT channel_id
              FROM logs.minute_broadcast
             WHERE channel_id IS NOT NULL
               AND time_utc > GETDATE() - interval '2 hour'
               AND (game <> 'watch parties' OR game IS NULL)
			   AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
        )

        SELECT recently_broadcast.channel_id,
               total_time_watched.total_minutes_watched
          FROM recently_broadcast
     LEFT JOIN total_time_watched
            ON recently_broadcast.channel_id = total_time_watched.channel_id
    `
	var results []*LifeTimeViewingDurationAggregate

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_minute_watched_lifetime.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*LifeTimeViewingDurationAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "redshift: failed to query lifetime viewing stats by channelID")
		}
	}

	c.stats.ExecutionTime("redshift.n_minute_watched_lifetime.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift: failed to close rows when selecting lifetime viewing stats by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		aggregate := &LifeTimeViewingDurationAggregate{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.MinutesWatched,
		)

		results = append(results, aggregate)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift: error scanning rows when selecting lifetime viewing stats by channelID"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
