package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NRaidConsecutiveBroadcast queries Redshift to count the number of raiding
// parties that exceed 2 raiders
func (c *Client) NRaidConsecutiveBroadcast(ctx context.Context) ([]*RaidsOverThreshold, error) {
	statement := `
        SELECT raid_source_id AS channel_id,
               COUNT(*) AS raids_count
          FROM spade.raid_execute
         WHERE viewer_count >= 2
           AND channel_id IS NOT NULL
      GROUP BY raid_source_id
      ORDER BY raids_count DESC
    `
	var results []*RaidsOverThreshold

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_raid_consecutive_broadcast.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*RaidsOverThreshold{}, nil
		default:
			return nil, errors.Wrap(err, "redshift: failed to query raids")
		}
	}

	c.stats.ExecutionTime("redshift.n_raid_consecutive_broadcast.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift: failed to close rows when selecting query raid stats"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		aggregate := &RaidsOverThreshold{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.RaidCount,
		)

		results = append(results, aggregate)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift: error scanning rows when selecting query raid stats"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
