package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NUniqueChatterBroadcast queries Redshift to count the max number of unique chatters
// when we split a broadcast into 10 min chunks
// This query only returns results for channels that have broadcast in the last 2 hours.
func (c *Client) NUniqueChatterBroadcast(ctx context.Context) ([]*UniqueChatterBroadcastAggregate, error) {
	statement := `
        WITH max_unique_chatters_per_channel AS
        (
            SELECT channel_id,
                   MAX(unique_chatters) AS max_unique_chatter
              FROM (
                SELECT scm.channel_id,
                       date_trunc('hour', "time") + 10 * (FLOOR(Date_part(minute, "time") / 10)) * interval '1 minutes' AS minute_group,
                       COUNT(DISTINCT user_id) AS unique_chatters
                  FROM spade.server_chat_message scm
                 WHERE scm.date >= $1 AND scm.date < $3
                   AND scm.time_utc > GETDATE() - interval '3 months'
                   AND scm.channel_id IS NOT NULL
              GROUP BY channel_id, minute_group
              )
            GROUP BY channel_id
        ),

        recently_broadcast AS
        (
            SELECT DISTINCT channel_id
                       FROM spade.minute_broadcast
                      WHERE date >= $2 AND date < $3
                        AND time_utc > getdate() - interval '2 hour'
                        AND channel_id IS NOT NULL
                        AND (game <> 'watch parties' OR game IS NULL)
						AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
        )

        SELECT max_unique_chatters_per_channel.channel_id,
               COALESCE(max_unique_chatters_per_channel.max_unique_chatter, 0) AS max_unique_chatter
          FROM recently_broadcast
     LEFT JOIN max_unique_chatters_per_channel
            ON recently_broadcast.channel_id = max_unique_chatters_per_channel.channel_id
    `
	var results []*UniqueChatterBroadcastAggregate
	currTime := time.Now()
	threeMonthsAgo := currTime.Truncate(24*time.Hour).AddDate(0, -3, -1).Format(sqlTimeFormat)
	oneDayAgo := currTime.Truncate(24*time.Hour).AddDate(0, 0, -2).Format(sqlTimeFormat)
	queryTimeRangeEnd := currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat)

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement, threeMonthsAgo, oneDayAgo, queryTimeRangeEnd)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_unique_chatter_broadcast.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*UniqueChatterBroadcastAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "redshift: failed to query unique chatter stats by channelID")
		}
	}

	c.stats.ExecutionTime("redshift.n_unique_chatter_broadcast.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift: failed to close rows when selecting unique chatter stats by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		aggregate := &UniqueChatterBroadcastAggregate{}
		err = rows.Scan(
			&aggregate.ChannelID,
			&aggregate.UniqueLogins,
		)

		if aggregate.ChannelID != "" {
			results = append(results, aggregate)
		}
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift: error scanning rows when selecting unique chatter stats by channelID"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
