package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// NViewers selects the max number of concurrent viewers for a channel
//
// This query does not run on a tahoe replica: The need for querying lifetime data is not well supported
// in a tahoe replica, and therefore we continue to use the achievements redshift cluster.
func (c *Client) NViewers(ctx context.Context) ([]*MaxConcurrentViewers, error) {
	statement := `
        WITH max_concurrents AS
        (
            SELECT channel_id,
                   MAX(total) AS viewer_count
              FROM logs.channel_concurrents
          GROUP BY channel_id
        ),

        recently_broadcast AS
        (
            SELECT DISTINCT channel_id
                       FROM logs.minute_broadcast
                      WHERE channel_id IS NOT NULL
                        AND time_utc > GETDATE() - interval '2 hour'
                        AND (game <> 'watch parties' OR game IS NULL)
						AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
        )

        SELECT recently_broadcast.channel_id,
               viewer_count
          FROM recently_broadcast
     LEFT JOIN max_concurrents
            ON recently_broadcast.channel_id = max_concurrents.channel_id
    `

	var results []*MaxConcurrentViewers

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.n_viewers_lifetime.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*MaxConcurrentViewers{}, nil
		default:
			return nil, errors.Wrap(err, "redshift: failed to query n viewers stats by channelID")
		}
	}

	c.stats.ExecutionTime("redshift.n_viewers_lifetime.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift: failed to close rows when selecting n viewers stats by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		viewers := &MaxConcurrentViewers{}
		err = rows.Scan(
			&viewers.ChannelID,
			&viewers.MaxViewers,
		)

		results = append(results, viewers)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift: error scanning rows when selecting n followers stats by channelID"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
