package redshift

import (
	"context"
	"database/sql"
	"time"

	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// QuestMetrics queries for the relevant quest stats.
// These quests use followers, minutes broadcast, unique days broadcast, and average ccu. We join all those
// stats together in this query.
// $1 = 6mo start time,
// $2 = 3mo start time,
// $3 = 30day start time
// $4 = end time (today + 1)
func (c *Client) QuestMetrics(ctx context.Context) ([]*QuestMetricAggregate, error) {
	statement := `
    WITH recent_follows AS
    (
      SELECT channel_id, follow_count
        FROM (
          SELECT target_user_id AS channel_id,
                 follow_count,
                 row_number() OVER (PARTITION BY target_user_id ORDER BY time_utc desc) AS rank
            FROM spade.server_follow
            WHERE follow_count IS NOT NULL
              AND channel_id IS NOT NULL
              AND date >= $1 AND date < $4
              AND time_utc > GETDATE() - interval '6 months'
        )
      WHERE rank = 1
    ),

    unique_chatters_per_channel AS
    (
      SELECT channel_id,
             MAX(unique_chatters) AS max_unique_chatter
        FROM (
          SELECT scm.channel_id,
                 date_trunc('hour', "time") + 10 * (FLOOR(Date_part(minute, "time") / 10)) * interval '1 minutes' AS minute_group,
                 COUNT(DISTINCT user_id) AS unique_chatters
            FROM spade.server_chat_message scm
           WHERE scm.date >= $1 AND scm.date < $4
             AND scm.channel_id IS NOT NULL
             AND scm.time_utc > GETDATE() - interval '6 months'
        GROUP BY channel_id, minute_group
        )
      GROUP BY channel_id
    ),

    sol_restriction_timeline AS
    (
      SELECT "time",
             is_restricted,
             lag(is_restricted,1) over (partition by channel_id order by "time" asc) as prev_restriction,
             lag("time",1) over (partition by channel_id order by "time" asc) as prev_time,
             lead(is_restricted,1) over (partition by channel_id order by "time" asc) as next_restriction,
             channel_id,
             toggle_method,
             restriction_type
        FROM spade.channel_restriction_status
       WHERE date >= $2 AND date < $4
         AND convert_timezone('US/Pacific', 'UTC', "time")  > GETDATE() - interval '30 day'
    ),

    sub_only_live AS
    (
        SELECT
                channel_id,
                case when next_restriction is null then convert_timezone('US/Pacific', 'UTC', "time")
                else convert_timezone('US/Pacific', 'UTC', prev_time) end as restriction_start_time,

                case when next_restriction is null then '2999-12-31 23:59:59'
                else convert_timezone('US/Pacific', 'UTC', "time") end as restriction_end_time,

                restriction_type,
                toggle_method as last_toggle_method
        FROM
                sol_restriction_timeline
        WHERE (
                  --updated CTE logic begins here
                  (is_restricted = false and prev_restriction = true and next_restriction is not null)
                OR
                  (next_restriction is null and is_restricted = true)
                  --additional logic for redundant toggle events for enabling restrictions
                OR
                  (is_restricted = true and prev_restriction = true)
                OR
                  (is_restricted = false and prev_restriction = true and next_restriction is null)
              )
              AND
                lower(restriction_type) = 'sub_only_live'
    ),

    raw_minutes_broadcast_without_sol AS
    (
         SELECT mb.channel_id,
                mb.time_utc,
                mb.broadcaster_software,
                date_trunc('minute', mb.time_utc) as minute_time_utc
           FROM spade.minute_broadcast mb
      LEFT JOIN sub_only_live sol
             ON sol.channel_id  = mb.channel_id
            AND mb.time_utc BETWEEN restriction_start_time AND restriction_end_time
          WHERE mb.date >= $3 AND mb.date < $4
            AND mb.channel_id IS NOT NULL
            AND mb.time_utc > GETDATE() - interval '30 day'
            AND (mb.game <> 'watch parties' OR mb.game IS NULL)
            AND sol.channel_id is null
      ),

     minutes_broadcast AS
     (
        SELECT mb.channel_id,
               AVG(cc.total::float) AS average_ccu,
               COUNT(mb.channel_id) AS mb_count,
               COUNT(DISTINCT date_part('day', mb.time_utc)) AS unique_days
          FROM raw_minutes_broadcast_without_sol mb
     FULL JOIN spade.channel_concurrents cc
            ON mb.channel_id = cc.channel_id
            AND minute_time_utc = date_trunc('minute', cc.time_utc)
            AND cc.date >= $3 AND cc.date < $4
            AND cc.content_mode <> 'prime_video_watch_party'
            WHERE mb.channel_id is not null
      GROUP BY mb.channel_id
     ),

     filtered_minutes_broadcast AS
     (
        SELECT mb.channel_id,
            COUNT(mb.channel_id) AS filtered_mb_count,
            COUNT(DISTINCT date_part('day', mb.time_utc)) AS filtered_unique_days
        FROM raw_minutes_broadcast_without_sol mb
        WHERE (mb.broadcaster_software IS NULL OR (mb.broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
        GROUP BY mb.channel_id
     ),

     mw_breakdown AS
     (
         SELECT channel_id,
                SUM(CASE WHEN (content = 'raid_channel' AND (host_channel = '' OR host_channel IS NULL)) THEN 1 ELSE 0 END) AS raided_mw,
                SUM(CASE WHEN (host_channel <> '' AND host_channel IS NOT NULL) THEN 1 ELSE 0 END) AS hosted_mw,
                COUNT(*) AS total_mw
           FROM spade."minute-watched"
          WHERE date >= $3 AND date < $4
            AND channel_id IS NOT NULL
            AND date_trunc('day', time_utc) > GETDATE() - interval '30 days'
            AND (live IS NULL OR live IS true)
            AND player <> 'embed'
            AND (broadcaster_software IS NULL OR (broadcaster_software NOT IN ('prime_video_watch_party', 'watch_party', 'watch_party_rerun', 'watch_party_premiere')))
            AND (channel_restriction_status is null or lower(channel_restriction_status) <> 'sub_only_live')
       GROUP BY channel_id
     )

    SELECT minutes_broadcast.channel_id,
           COALESCE(recent_follows.follow_count, 0) AS follow_count,
           COALESCE(mw_breakdown.raided_mw, 0) AS raided_mw,
           COALESCE(mw_breakdown.hosted_mw, 0) AS hosted_mw,
           COALESCE(mw_breakdown.total_mw, 0) AS total_mw,
           COALESCE(minutes_broadcast.mb_count, 0) AS mb_count,
           COALESCE(minutes_broadcast.unique_days, 0) AS unique_days,
           COALESCE(minutes_broadcast.average_ccu, 0) AS average_ccu,
           COALESCE(unique_chatters_per_channel.max_unique_chatter, 0) AS unique_chatters,
           COALESCE(filtered_minutes_broadcast.filtered_mb_count, 0) AS filtered_mb_count,
           COALESCE(filtered_minutes_broadcast.filtered_unique_days, 0) AS filtered_unique_days
      FROM minutes_broadcast
 LEFT JOIN recent_follows
        ON (recent_follows.channel_id = minutes_broadcast.channel_id)
 LEFT JOIN mw_breakdown
        ON (minutes_broadcast.channel_id = mw_breakdown.channel_id)
 LEFT JOIN unique_chatters_per_channel
        ON (minutes_broadcast.channel_id = unique_chatters_per_channel.channel_id)
 LEFT JOIN filtered_minutes_broadcast
        ON (minutes_broadcast.channel_id = filtered_minutes_broadcast.channel_id)
`

	var resultsAggregate []*QuestMetricAggregate
	currTime := time.Now()

	// These times are represented in PST, but we care about UTC times for actual data. We pad the
	// edges of the ranges so that all the utc data we care about is still inside these ranges.
	sixMonthsAgo := currTime.Truncate(24*time.Hour).AddDate(0, -6, -1).Format(sqlTimeFormat)
	threeMonthsAgo := currTime.Truncate(24*time.Hour).AddDate(0, -3, -1).Format(sqlTimeFormat)
	thirtyDaysAgo := currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat)
	queryTimeRangeEnd := currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat)

	start := time.Now()
	rows, err := c.db.QueryContext(ctx, statement, sixMonthsAgo, threeMonthsAgo, thirtyDaysAgo, queryTimeRangeEnd)
	elapsed := time.Since(start)

	if err != nil {
		c.stats.ExecutionTime("redshift.quest_metrics.error", elapsed)

		switch err {
		case sql.ErrNoRows:
			return []*QuestMetricAggregate{}, nil
		default:
			return nil, errors.Wrap(err, "pq: failed to query for quest progress")
		}
	}

	c.stats.ExecutionTime("redshift.quest_metrics.success", elapsed)

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement": statement,
			}).Error("pq: failed to close rows")
		}
	}()

	for rows.Next() {
		questAggregate := &QuestMetricAggregate{}
		err = rows.Scan(
			&questAggregate.ChannelID,
			&questAggregate.FollowCount,
			&questAggregate.RaidedMinuteWatched,
			&questAggregate.HostedMinuteWatched,
			&questAggregate.TotalMinuteWatched,
			&questAggregate.MinuteBroadcastCount,
			&questAggregate.UniqueDays,
			&questAggregate.AverageCCU,
			&questAggregate.UniqueChatters,
			&questAggregate.FilteredMinuteBroadcastCount,
			&questAggregate.FilteredUniqueDays,
		)

		if err != nil {
			log.WithError(err).WithField("quest_aggregate", questAggregate).Error("pq: failed to scan quest metric")
			continue
		}

		var hostRaidPct float64
		if questAggregate.TotalMinuteWatched > 0 {
			hostRaidPct = float64(questAggregate.RaidedMinuteWatched+questAggregate.HostedMinuteWatched) / float64(questAggregate.TotalMinuteWatched)
		}

		questAggregate.LiveOnlyAverageCCU = (1 - hostRaidPct) * questAggregate.AverageCCU

		resultsAggregate = append(resultsAggregate, questAggregate)
	}

	err = rows.Err()
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"statement": statement,
		}).Error("pq: failed to scan rows")

		return nil, err
	}

	return resultsAggregate, nil
}
