package redshift

import (
	"context"
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type questMetrics struct {
	suite.Suite
	redshiftClient    *Client
	mock              sqlmock.Sqlmock
	sixMonthsAgo      string
	threeMonthsAgo    string
	thirtyDaysAgo     string
	queryTimeRangeEnd string
}

func TestQuestMetrics(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)
	currTime := time.Now()

	s := &questMetrics{
		redshiftClient:    &Client{stub, stats.NewNoopClient()},
		mock:              mock,
		sixMonthsAgo:      currTime.Truncate(24*time.Hour).AddDate(0, -6, -1).Format(sqlTimeFormat),
		threeMonthsAgo:    currTime.Truncate(24*time.Hour).AddDate(0, -3, -1).Format(sqlTimeFormat),
		thirtyDaysAgo:     currTime.Truncate(24*time.Hour).AddDate(0, 0, -31).Format(sqlTimeFormat),
		queryTimeRangeEnd: currTime.Truncate(24*time.Hour).AddDate(0, 0, 1).Format(sqlTimeFormat),
	}

	suite.Run(t, s)
}

func (s *questMetrics) TestNoRows_ReturnsEmptyArray() {
	s.mock.ExpectQuery("SELECT minutes_broadcast.channel_id").
		WithArgs(s.sixMonthsAgo, s.threeMonthsAgo, s.thirtyDaysAgo, s.queryTimeRangeEnd).
		WillReturnError(sql.ErrNoRows)

	results, _ := s.redshiftClient.QuestMetrics(context.Background())
	s.Equal(results, []*QuestMetricAggregate{})

	err := s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *questMetrics) TestUnexpectedError() {
	s.mock.ExpectQuery("SELECT minutes_broadcast.channel_id").
		WithArgs(s.sixMonthsAgo, s.threeMonthsAgo, s.thirtyDaysAgo, s.queryTimeRangeEnd).
		WillReturnError(sql.ErrTxDone)

	results, err := s.redshiftClient.QuestMetrics(context.Background())
	s.Error(err)
	s.Nil(results)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *questMetrics) TestSuccess() {
	s.mock.ExpectQuery("SELECT minutes_broadcast.channel_id").
		WithArgs(s.sixMonthsAgo, s.threeMonthsAgo, s.thirtyDaysAgo, s.queryTimeRangeEnd).
		WillReturnRows(
			sqlmock.NewRows([]string{
				"channel_id",
				"follow_count",
				"raided_mw",
				"hosted_mw",
				"total_mw",
				"mb_count",
				"unique_days",
				"average_ccu",
				"max_unique_chatter",
				"filtered_mb_count",
				"filtered_unique_days",
			}).
				AddRow("123456",
					"500",
					"25",
					"25",
					"250",
					"75",
					"12",
					"10.0",
					"3",
					"75",
					"12"),
		)

	results, err := s.redshiftClient.QuestMetrics(context.Background())
	s.NoError(err)

	s.Equal(1, len(results))

	if len(results) > 0 {
		s.Equal("123456", results[0].ChannelID)
		s.Equal(500, results[0].FollowCount)
		s.Equal(25, results[0].RaidedMinuteWatched)
		s.Equal(25, results[0].HostedMinuteWatched)
		s.Equal(250, results[0].TotalMinuteWatched)
		s.Equal(75, results[0].MinuteBroadcastCount)
		s.Equal(12, results[0].UniqueDays)
		s.Equal(10.0, results[0].AverageCCU)
		s.Equal(8.0, results[0].LiveOnlyAverageCCU)
		s.Equal(3, results[0].UniqueChatters)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *questMetrics) TestFailure_WithEmptyFields() {
	s.mock.ExpectQuery("SELECT minutes_broadcast.channel_id").
		WithArgs(s.sixMonthsAgo, s.threeMonthsAgo, s.thirtyDaysAgo, s.queryTimeRangeEnd).
		WillReturnRows(
			sqlmock.NewRows([]string{
				"channel_id",
				"follow_count",
				"raided_mw",
				"hosted_mw",
				"total_mw",
				"mb_count",
				"unique_days",
				"average_ccu",
				"max_unique_chatter",
				"filtered_mb_count",
				"filtered_unique_days",
			}).
				AddRow("123456",
					"500",
					"0",
					"25",
					nil,
					nil,
					nil,
					nil,
					"10",
					"75",
					nil),
		)

	results, _ := s.redshiftClient.QuestMetrics(context.Background())
	s.Nil(results)
}
