package redshift

import (
	"database/sql"
	"fmt"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	log "github.com/sirupsen/logrus"
)

const (
	driver             = "postgres"
	dataSourceTemplate = "sslmode=require user=%s password=%s host=%s port=%s dbname=%s"
	sqlTimeFormat      = "2006-01-02 15:04:05"
)

// Client contains a db client and a stats client for timing redshift queries
type Client struct {
	db    *sql.DB
	stats stats.StatsdClient
}

// Config are the credentials needed to connect to redshift
type Config struct {
	Host     string
	User     string
	Password string
	Port     string
	DBName   string
}

// NewClient opens a connection pool to the PostgresSQL database,
// specified by the database credentials and address.
//
// Pinging the database ensures that the connection was established.
func NewClient(creds *Config, statsClient stats.StatsdClient) (*Client, error) {
	connection := fmt.Sprintf(dataSourceTemplate, creds.User, creds.Password, creds.Host, creds.Port, creds.DBName)
	log.Info("Opening connection to Postgres database at ", creds.Host)

	db, err := sql.Open(driver, connection)
	if err != nil {
		return nil, err
	}

	if err := db.Ping(); err != nil {
		return nil, err
	}

	return &Client{
		db:    db,
		stats: statsClient,
	}, nil
}
