package sandstorm

import (
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// Manager wraps the sandstorm manager
type Manager struct {
	*manager.Manager
}

// NewManager returns a new sandstorm manager
func NewManager(roleARN string, region string) (*Manager, error) {
	awsConfig := aws.NewConfig().WithRegion(region).WithSTSRegionalEndpoint(endpoints.RegionalSTSEndpoint)

	sess, err := session.NewSession(awsConfig)
	if err != nil {
		return &Manager{}, errors.Wrap(err, "sandstorm: failed to initialize new aws session")
	}

	creds := credentials.NewCredentials(&stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      roleARN,
		Client:       sts.New(sess),
	})

	sandstormManager := manager.New(manager.Config{
		AWSConfig:   awsConfig.WithCredentials(creds),
		ServiceName: "achievements",
	})

	log.Info("Initialized sandstorm manager for role ", roleARN)

	return &Manager{
		sandstormManager,
	}, nil
}

// GetSecret returns a secret with the given name
func (m *Manager) GetSecret(secretName string) (string, error) {
	secret, err := m.Get(secretName)
	if err != nil {
		return "", errors.Wrap(err, "sandstorm: failed to get secret")
	}

	return string(secret.Plaintext), nil
}
