package sns

import (
	"context"
	"encoding/json"
	"fmt"

	"code.justin.tv/cb/achievements/internal/awscredentials"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/pkg/errors"
)

// Client is a wrapper for the snsClinet.
type Client struct {
	snsClient *sns.SNS
	topic     string
}

// NewClient creates an instance of an sns client
func NewClient(env string, region string, topic string) (*Client, error) {
	sess, err := session.NewSession(&aws.Config{
		Credentials: awscredentials.New(env, region),
		MaxRetries:  aws.Int(3),
		Region:      aws.String(region),
	})

	if err != nil {
		return nil, errors.Wrap(err, "sns: failed to instantiate new session")
	}

	return &Client{
		snsClient: sns.New(sess),
		topic:     topic,
	}, nil
}

// PublishNotification sends a message to the specified topic
func (c *Client) PublishNotification(ctx context.Context, message interface{}) error {
	messageString, err := jsonMarshalToString(message)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("sns: failed to json marshal message: %s", message))
	}

	input := &sns.PublishInput{
		TopicArn: aws.String(c.topic),
		Message:  aws.String(messageString),
	}

	err = input.Validate()
	if err != nil {
		msg := fmt.Sprintf("sns: invalid publish input with message: %s", messageString)
		return errors.Wrap(err, msg)
	}

	_, err = c.snsClient.PublishWithContext(ctx, input)
	if err != nil {
		msg := fmt.Sprintf("sns: failed to send to topic '%s' for message: %s", c.topic, messageString)
		return errors.Wrap(err, msg)
	}

	return nil
}

func jsonMarshalToString(message interface{}) (string, error) {
	messageBytes, err := json.Marshal(message)
	if err != nil {
		return "", err
	}

	return string(messageBytes), nil
}
