package sqs

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/achievements/internal/awscredentials"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/pkg/errors"
)

// MessageAttributeKeyAchievementKey is a custom message attribute,
// included in every message from the Sourcer to the Worker's SQS,
// that specifies the Achievement for the Worker.
const MessageAttributeKeyAchievementKey = "AchievementKey"

// Client is a wrapper for the sqs.SQS client for a specific queue.
type Client struct {
	sqs      *sqs.SQS
	queueURL string
}

// NewClient creates an instance of an SQS Client
func NewClient(env string, region string, queueURL string) (*Client, error) {
	sess, err := session.NewSession(&aws.Config{
		Credentials: awscredentials.New(env, region),
		HTTPClient: &http.Client{
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 200,
			},
		},
		MaxRetries: aws.Int(3),
		Region:     aws.String(region),
	})

	if err != nil {
		return nil, errors.Wrap(err, "sqs: failed to initialize client")
	}

	return &Client{
		sqs:      sqs.New(sess),
		queueURL: queueURL,
	}, nil
}

// Send sends a message to the specified topic
func (c *Client) Send(ctx context.Context, achievementKey string, message interface{}) error {
	achievementAttributeValue, err := messageAttributeValueString(achievementKey)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for achievement key: %s", achievementKey)
		return errors.Wrap(err, msg)
	}

	messageString, err := jsonMarshalToString(message)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("sqs: failed to json marshal message: %s", message))
	}

	messageAttributes := map[string]*sqs.MessageAttributeValue{
		MessageAttributeKeyAchievementKey: achievementAttributeValue,
	}

	input, err := c.newSendMessageInput(messageAttributes, messageString)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("sqs: invalid send message input with message: %s", messageString))
	}

	_, err = c.sqs.SendMessageWithContext(ctx, input)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("sqs: failed to send message: %s", messageString))
	}

	return nil
}

const messageAttributeValueDataTypeString = "String"

func messageAttributeValueString(str string) (*sqs.MessageAttributeValue, error) {
	attribute := &sqs.MessageAttributeValue{}

	attribute.SetDataType(messageAttributeValueDataTypeString)
	attribute.SetStringValue(str)

	if err := attribute.Validate(); err != nil {
		return nil, err
	}

	return attribute, nil
}

func jsonMarshalToString(message interface{}) (string, error) {
	messageBytes, err := json.Marshal(message)
	if err != nil {
		return "", err
	}

	return string(messageBytes), nil
}

func (c *Client) newSendMessageInput(attrs map[string]*sqs.MessageAttributeValue, msgBody string) (*sqs.SendMessageInput, error) {
	input := &sqs.SendMessageInput{}

	input.SetQueueUrl(c.queueURL)
	input.SetMessageAttributes(attrs)
	input.SetMessageBody(msgBody)

	if err := input.Validate(); err != nil {
		return nil, err
	}

	return input, nil
}
