package stats

import (
	"fmt"
	"time"

	"code.justin.tv/common/gometrics"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// Client is the statsd client
type Client struct {
	statsd.Statter
}

const (
	flushInterval        = 1 * time.Second
	gometricsMonitorRate = 1 * time.Second
)

// NewClient instatiates a new statsd client
func NewClient(host, env, name string) (*Client, error) {
	prefix := fmt.Sprintf("cb.achievements.%v.%v", name, env)

	statter, err := statsd.NewBufferedClient(host, prefix, flushInterval, 512)
	if err != nil {
		return nil, errors.Wrap(err, "statsd: failed to instantiate buffered client")
	}

	log.Info(fmt.Sprintf("Connected to StatsD at %s with prefix %s", host, prefix))

	gometrics.Monitor(statter, gometricsMonitorRate)

	return &Client{
		Statter: statter,
	}, nil
}

// ExecutionTime records the execution time given a duration. This is intended
// to be used outside of the request middleware, for query/method timing.
// This can also be called in a goroutine for performance reasons.
func (c *Client) ExecutionTime(statName string, duration time.Duration) {
	err := c.TimingDuration(statName, duration, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"duration":  duration,
		}).Warn("statsd: failed to send timing duration")
	}
}

// SendGauge sends a gauge value to statsd. This is wrapped in the client for brevity,
// because error handling is required but takes up lots of space for the caller.
func (c *Client) SendGauge(statName string, value int64) {
	err := c.Gauge(statName, value, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"value":     value,
		}).Warn("statsd: failed to send gauge")
	}
}
