package tracking

import (
	"fmt"
	"time"

	"code.justin.tv/hygienic/spade"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/cb/achievements/config"
	"code.justin.tv/cb/achievements/env"
	"code.justin.tv/cb/achievements/internal/clients/db"
)

// Client is the wrapper for the spade client
type Client struct {
	spade *spade.Client
}

// NewClient creates a new instance of the tracking client, and
// starts up all the workers needed in the worker pool
func NewClient() (*Client, error) {
	spadeClient := &spade.Client{
		Config: &spade.StaticConfig{
			Concurrency:         SpadeParallelWorkersCount,
			BatchSize:           SpadeEventBufferSize,
			CombineSmallBatches: true,
		},
	}

	err := spadeClient.Setup()
	if err != nil {
		return nil, err
	}

	// Runs a background process to drain the client's queue of events
	go func() {
		err := spadeClient.Start()
		if err != nil {
			log.WithError(err).Fatal("error starting spade client")
		}
	}()

	return &Client{
		spade: spadeClient,
	}, nil
}

// Close closes the events channel and blocks until all workers complete.
func (c *Client) Close() {
	c.spade.Close()
}

// TrackCompletedAchievements will iterate over the progressions array and
// feed the events into the channel
func (c *Client) TrackCompletedAchievements(progressions []db.Progression) {
	now := time.Now()
	eventName := AchievementCompleted

	if config.Environment != env.Production {
		eventName += fmt.Sprintf("-%s", config.Environment)
	}

	events := make([]spade.Event, len(progressions))
	for _, progression := range progressions {
		customEvent := SpadeEvent{
			Name:            eventName,
			ServerTimestamp: float64(now.Unix()),
			ChannelID:       progression.ChannelID,
			AchievementID:   progression.AchievementID,
			AchievementName: progression.AchievementKey,
			Level:           progression.AchievementLevel,
		}

		event := spade.Event{
			Name:       eventName,
			Properties: customEvent,
		}
		events = append(events, event)
	}

	c.spade.QueueEvents(events...)
}
