package twitchcon

import (
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/pkg/errors"
)

// TicketPurchaser handles methods for fetching data related to twitchcon purchases
type TicketPurchaser interface {
	GetPurchasers() ([]string, error)
}

// Client wraps an http client and implements the TwitchconUserFectcher interface
type Client struct {
	PurchasersURL string
	BearerToken   string
}

// PurchasersResp is the expected resp from the purchasers endpoint
type PurchasersResp struct {
	Attendees struct {
		Data  []string
		Count int
	}
}

// NewClient creates a new twitchcon client
func NewClient(url, bearerToken string) *Client {
	return &Client{
		PurchasersURL: url,
		BearerToken:   bearerToken,
	}
}

// GetPurchasers returns a list of channel ids that have purchased a twithcon ticket for the current twitchcon event.
// The config values related to twitchcon need to be updated to the next twitchcon in order for this method
// to be up to date with the most recent purchasers.
func (c *Client) GetPurchasers() ([]string, error) {
	httpClient := &http.Client{}
	req, err := http.NewRequest("GET", c.PurchasersURL, nil)
	if err != nil {
		return nil, errors.Wrap(err, "twitchcon: failed to create http request")
	}

	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", c.BearerToken))

	resp, err := httpClient.Do(req)
	if err != nil {
		return nil, errors.Wrap(err, "twitchcon: call to purchasers url failed")
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("twitchcon: call to purchasers url failed with '%s'", resp.Status)
	}

	defer resp.Body.Close()
	body := &PurchasersResp{}
	err = json.NewDecoder(resp.Body).Decode(body)
	if err != nil {
		return nil, errors.Wrap(err, "twitchcon: failed to decode json body")
	}

	if body.Attendees.Count == 0 {
		return nil, errors.New("twitchcon: failed to decode body into json, count returned 0")
	}

	if len(body.Attendees.Data) == 0 {
		return nil, errors.New("twitchcon: failed to decode body into json, data slice is empty")
	}

	return body.Attendees.Data, nil
}
