package sourcer

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"

	"code.justin.tv/cb/achievements/config"

	achievement "code.justin.tv/cb/achievements/internal/achievement/firststream"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type firstStreamInputMessage struct {
	ChannelID int `json:"channel_id"`
}

func (s *Server) processFirstStreamEvent(ctx context.Context, input string) error {
	table := config.Values.DynamoDB.Tables.FirstStreams
	message := &firstStreamInputMessage{}

	err := json.Unmarshal([]byte(input), message)
	if err != nil {
		msg := fmt.Sprintf("%s error: failed to parse input message", achievement.Key)
		return errors.Wrap(err, msg)
	}

	if message.ChannelID == 0 {
		log.WithFields(log.Fields{
			"achievement_key": achievement.Key,
			"message":         input,
		}).Info(fmt.Sprintf("%s: received message with invalid channel id", achievement.Key))

		return nil
	}

	channelID := strconv.Itoa(message.ChannelID)

	isDuplicate, err := s.dynamoDB.TableHasChannel(ctx, table, channelID)
	if err != nil {
		msg := fmt.Sprintf("%s error: failed to dedupe input message", achievement.Key)
		return errors.Wrap(err, msg)
	}
	if isDuplicate {
		return nil
	}

	err = s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		ChannelID: channelID,
	})

	go s.SaveSQSState(achievement.Key, 1)

	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("%s error: failed to send to worker sqs", achievement.Key))
	}

	go func() {
		err = s.dynamoDB.PersistChannelToTable(context.Background(), table, channelID)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"achievement_key": achievement.Key,
				"message":         input,
			}).Error(fmt.Sprintf("%s error: failed to put to dynamodb '%s'", achievement.Key, table))
		}
	}()

	return nil
}
