package sourcer

import (
	"context"
	"fmt"
	"math"
	"net/http"

	"code.justin.tv/cb/achievements/internal/clients/dynamo"

	achievement "code.justin.tv/cb/achievements/internal/achievement/ndaysnoviewers"
	log "github.com/sirupsen/logrus"
)

// nDaysNoViewers queries redshift and breaks the results into batches of size
// 	SNSBatchSize then sends the batch to the worker via SNS
func (s *Server) nDaysNoViewers(w http.ResponseWriter, r *http.Request) {
	featureEnabled, err := s.dynamoDB.IsFeatureEnabled(context.Background(), dynamo.NDaysNoViewersFeature)
	if err != nil {
		log.WithError(err).Error("failed to read featuredEnabled flag")
		return
	}

	if !featureEnabled {
		return
	}

	channels, err := s.redshift.NDaysNoViewers(r.Context())
	if err != nil {
		msg := fmt.Sprintf("cron: %s redshift query failed", achievement.Key)

		log.WithError(err).Error(msg)
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(channels)) / float64(sqsBatchSize))
	go s.SaveSQSState(achievement.Key, int64(batchCount))

	for start := 0; start < len(channels); start += sqsBatchSize {
		end := start + sqsBatchSize
		if end > len(channels) {
			end = len(channels)
		}

		batch := make([]*achievement.ChannelConsecutiveSessionsNoViewers, end-start)

		for idx, result := range channels[start:end] {
			batch[idx] = &achievement.ChannelConsecutiveSessionsNoViewers{
				ChannelID:           result.ChannelID,
				ConsecutiveSessions: result.ConsecutiveSessions,
			}
		}

		go s.sendNDaysNoViewers(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNDaysNoViewers(ctx context.Context, batch []*achievement.ChannelConsecutiveSessionsNoViewers) {
	err := s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		Channels: batch,
	})

	if err != nil {
		msg := fmt.Sprintf("cron: %s failed to send batch to worker sqs", achievement.Key)

		log.WithError(err).Error(msg)
	}
}
