package sourcer

import (
	"context"
	"math"
	"net/http"

	achievement "code.justin.tv/cb/achievements/internal/achievement/nfollowers"
	log "github.com/sirupsen/logrus"
)

// nFollowers queries the last 6 months and returns the last known follower count for
// the user. This is meant to be run daily to refresh ea user's checkmark
func (s *Server) nFollowers(w http.ResponseWriter, r *http.Request) {
	followers, err := s.dirtyTahoeReplica.NFollowers(r.Context(), 0)
	if err != nil {
		log.WithError(err).Error("Failed to query redshift for n_followers")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(followers)) / float64(sqsBatchSize))
	go s.SaveSQSState(achievement.Key, int64(batchCount))

	for startIndex := 0; startIndex < len(followers); startIndex += sqsBatchSize {
		endIndex := startIndex + sqsBatchSize
		if endIndex > len(followers) {
			endIndex = len(followers)
		}

		batch := make([]*achievement.ChannelFollowers, endIndex-startIndex)

		for idx, result := range followers[startIndex:endIndex] {
			batch[idx] = &achievement.ChannelFollowers{
				ChannelID:   result.ChannelID,
				FollowCount: result.Followers,
			}
		}

		go s.sendNFollowers(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNFollowers(ctx context.Context, batch []*achievement.ChannelFollowers) {
	err := s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		Channels: batch,
	})

	if err != nil {
		log.WithError(err).Error("failed to send n_followers event to worker sqs")
	}
}
