package sourcer

import (
	"context"
	"math"
	"net/http"

	achievement "code.justin.tv/cb/achievements/internal/achievement/broadcastnhours30days"
	log "github.com/sirupsen/logrus"
)

// nHours30Days queries redshift and aggregates broadcast-minutes
// in the last 30 days for channels that have broadcast in the last 12 hours,
// and (TODO) sends those results to SNS to be processed by the achievements engine
// to update progress=.
func (s *Server) nHours30Days(w http.ResponseWriter, r *http.Request) {
	broadcasts, err := s.dirtyTahoeReplica.BroadcastNHours30Days(r.Context(), 0)
	if err != nil {
		log.WithError(err).Error("Failed to query redshift for broadcast_n_hours_30_days")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(broadcasts)) / float64(sqsBatchSize))
	go s.SaveSQSState(achievement.Key, int64(batchCount))

	for startIndex := 0; startIndex < len(broadcasts); startIndex += sqsBatchSize {
		endIndex := startIndex + sqsBatchSize
		if endIndex > len(broadcasts) {
			endIndex = len(broadcasts)
		}

		batch := make([]*achievement.Broadcasts, endIndex-startIndex)

		for idx, result := range broadcasts[startIndex:endIndex] {
			batch[idx] = &achievement.Broadcasts{
				ChannelID: result.ChannelID,
				Minutes:   result.Minutes,
			}
		}

		go s.sendNHours30Days(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNHours30Days(ctx context.Context, broadcasts []*achievement.Broadcasts) {
	err := s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		Broadcasts: broadcasts,
	})

	if err != nil {
		log.WithError(err).Error("failed to send broadcast_n_hours_30_days event to worker sqs")
	}
}
