package sourcer

import (
	"context"
	"fmt"
	"math"
	"net/http"

	achievement "code.justin.tv/cb/achievements/internal/achievement/nhourslifetime"
	log "github.com/sirupsen/logrus"
)

// nHoursLifetime queries redshift and breaks the results into batches of size
// 	SNSBatchSize then sends the batch to the worker via SNS
func (s *Server) nHoursLifetime(w http.ResponseWriter, r *http.Request) {
	channels, err := s.redshift.NHoursLifetime(r.Context())
	if err != nil {
		msg := fmt.Sprintf("cron: %s redshift query failed", achievement.Key)

		log.WithError(err).Error(msg)
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(channels)) / float64(sqsBatchSize))
	go s.SaveSQSState(achievement.Key, int64(batchCount))

	for start := 0; start < len(channels); start += sqsBatchSize {
		end := start + sqsBatchSize
		if end > len(channels) {
			end = len(channels)
		}

		batch := make([]*achievement.ChannelMinutesBroadcast, end-start)

		for idx, result := range channels[start:end] {
			batch[idx] = &achievement.ChannelMinutesBroadcast{
				ChannelID:        result.ChannelID,
				MinutesBroadcast: result.MinutesBroadcast,
			}
		}

		go s.sendNHoursLifetime(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNHoursLifetime(ctx context.Context, batch []*achievement.ChannelMinutesBroadcast) {
	err := s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		Channels: batch,
	})

	if err != nil {
		msg := fmt.Sprintf("cron: %s failed to send batch to worker sqs", achievement.Key)

		log.WithError(err).Error(msg)
	}
}
