package sourcer

import (
	"context"
	"fmt"
	"math"
	"net/http"

	achievement "code.justin.tv/cb/achievements/internal/achievement/nminutewatchedlifetime"
	log "github.com/sirupsen/logrus"
)

// nMinuteWatchedLifetime queries redshift and breaks the results into batches of size
// 	SNSBatchSize then sends the batch to the worker via SNS
func (s *Server) nMinuteWatchedLifetime(w http.ResponseWriter, r *http.Request) {
	channels, err := s.redshift.NMinuteWatchedLifetime(r.Context())
	if err != nil {
		msg := fmt.Sprintf("cron: %s redshift query failed", achievement.Key)

		log.WithError(err).Error(msg)
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(channels)) / float64(sqsBatchSize))
	go s.SaveSQSState(achievement.Key, int64(batchCount))

	for start := 0; start < len(channels); start += sqsBatchSize {
		end := start + sqsBatchSize
		if end > len(channels) {
			end = len(channels)
		}

		batch := make([]*achievement.ChannelMinutesWatched, end-start)

		for idx, result := range channels[start:end] {
			batch[idx] = &achievement.ChannelMinutesWatched{
				ChannelID:      result.ChannelID,
				MinutesWatched: result.MinutesWatched,
			}
		}

		go s.sendNMinuteWatchedLifetime(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNMinuteWatchedLifetime(ctx context.Context, batch []*achievement.ChannelMinutesWatched) {
	err := s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		Channels: batch,
	})

	if err != nil {
		log.WithError(err).Error(fmt.Sprintf("cron: %s failed to send batch to worker sqs", achievement.Key))
	}
}
