package sourcer

import (
	"context"
	"fmt"
	"math"
	"net/http"

	chatters "code.justin.tv/cb/achievements/internal/achievement/nuniquechatterbroadcast"
	"code.justin.tv/cb/achievements/internal/achievement/talkalittle"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	log "github.com/sirupsen/logrus"
)

// nUniqueChatterBroadcast queries redshift and breaks the results into batches of size
// SNSBatchSize then sends the batch to the worker via SNS
//
// We also use this redshift query to populate the metrics for the Talk a Little achievement as well,
// rather than duplicate the effort. We send the same results with a different sqs header (another
// achievement key) so that the worker can process that achievement differently.
func (s *Server) nUniqueChatterBroadcast(w http.ResponseWriter, r *http.Request) {
	channels, err := s.dirtyTahoeReplica.NUniqueChatterBroadcast(r.Context())
	if err != nil {
		msg := fmt.Sprintf("cron: %s redshift query failed", chatters.Key)

		log.WithError(err).Error(msg)
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(channels)) / float64(sqsBatchSize))
	go s.SaveSQSState(chatters.Key, int64(batchCount))
	go s.SaveSQSState(talkalittle.Key, int64(batchCount))

	for start := 0; start < len(channels); start += sqsBatchSize {
		end := start + sqsBatchSize
		if end > len(channels) {
			end = len(channels)
		}

		batch := make([]*chatters.ChannelUniqueChatters, end-start)

		for idx, result := range channels[start:end] {
			batch[idx] = &chatters.ChannelUniqueChatters{
				ChannelID:    result.ChannelID,
				UniqueLogins: result.UniqueLogins,
			}
		}

		go s.sendNUniqueChatterBroadcast(context.Background(), batch)
		go s.sendTalkALittle(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNUniqueChatterBroadcast(ctx context.Context, batch []*chatters.ChannelUniqueChatters) {
	err := s.sqs.Send(ctx, chatters.Key, &chatters.Input{
		Channels: batch,
	})
	if err != nil {
		msg := fmt.Sprintf("cron: %s failed to send batch to worker sqs", chatters.Key)
		log.WithError(err).Error(msg)
	}
}

func (s *Server) sendTalkALittle(ctx context.Context, batch []*chatters.ChannelUniqueChatters) {
	featureEnabled, err := s.dynamoDB.IsFeatureEnabled(context.Background(), dynamo.TalkALittleFeature)
	if err != nil {
		log.WithError(err).Error("failed to read featuredEnabled flag")
		return
	}

	if !featureEnabled {
		return
	}

	err = s.sqs.Send(ctx, talkalittle.Key, &chatters.Input{
		Channels: batch,
	})
	if err != nil {
		msg := fmt.Sprintf("cron: %s failed to send batch to worker sqs", talkalittle.Key)
		log.WithError(err).Error(msg)
	}
}
