package sourcer

import (
	"context"
	"fmt"
	"math"
	"net/http"

	achievement "code.justin.tv/cb/achievements/internal/achievement/nviewerslifetime"
	log "github.com/sirupsen/logrus"
)

// nViewers queries redshift for concurrent viewers over a time period of 1.5 hours.
func (s *Server) nViewers(w http.ResponseWriter, r *http.Request) {
	channels, err := s.redshift.NViewers(r.Context())
	if err != nil {
		msg := fmt.Sprintf("cron: %s redshift query failed", achievement.Key)

		log.WithError(err).Error(msg)
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	batchCount := math.Ceil(float64(len(channels)) / float64(sqsBatchSize))
	go s.SaveSQSState(achievement.Key, int64(batchCount))

	for start := 0; start < len(channels); start += sqsBatchSize {
		end := start + sqsBatchSize
		if end > len(channels) {
			end = len(channels)
		}

		batch := make([]*achievement.ChannelMaxConcurrentViewers, end-start)

		for idx, result := range channels[start:end] {
			batch[idx] = &achievement.ChannelMaxConcurrentViewers{
				ChannelID:  result.ChannelID,
				MaxViewers: result.MaxViewers,
			}
		}

		go s.sendNViewers(context.Background(), batch)
	}

	w.WriteHeader(http.StatusOK)
}

func (s *Server) sendNViewers(ctx context.Context, batch []*achievement.ChannelMaxConcurrentViewers) {
	err := s.sqs.Send(ctx, achievement.Key, &achievement.Input{
		Channels: batch,
	})

	if err != nil {
		msg := fmt.Sprintf("cron: %s failed to send batch to worker sqs", achievement.Key)

		log.WithError(err).Error(msg)
	}
}
