package sourcer

import (
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/achievements/config"
	log "github.com/sirupsen/logrus"
)

// RequestBody is the format of the SQS Message posted to the worker
type RequestBody struct {
	Type     string
	Message  string
	TopicArn string
}

// process is the handler for the /process endpoint which will handle
// 	get all incoming sqs events and based on sns arn process the events
func (s *Server) process(w http.ResponseWriter, req *http.Request) {
	var input RequestBody

	// get the arn from the event
	err := json.NewDecoder(req.Body).Decode(&input)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	switch input.TopicArn {
	case config.Values.AchievementSNSTopics.FirstStream:
		err = s.processFirstStreamEvent(req.Context(), input.Message)
	case config.Values.AchievementSNSTopics.StreamManagerVisited:
		err = s.processSingleDashboardVisitEvent(req.Context(), input.Message)
	case config.Values.AchievementSNSTopics.NAutohost:
		err = s.processNAutohostEvent(req.Context(), input.Message)
	case config.Values.AchievementSNSTopics.ChannelUpdate:
		err = s.processChannelUpdateEvent(req.Context(), input.Message)
		break
	default:
		log.WithFields(log.Fields{
			"message":   input.Message,
			"topic_arn": input.TopicArn,
			"type":      input.Type,
		}).Info(fmt.Sprintf("app: unhandled topic %s", input.TopicArn))
	}

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"message":   input.Message,
			"topic_arn": input.TopicArn,
			"type":      input.Type,
		}).Error("app: failed to process event")

		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	w.WriteHeader(http.StatusOK)
}
