package sourcer

import (
	"fmt"
	"net/http"

	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/redshift"
	"code.justin.tv/cb/achievements/internal/clients/sqs"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/clients/twitchcon"
	"code.justin.tv/cb/achievements/internal/httputil"
	log "github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

// sqsBatchSize is the max amount of channels the sourcer will include in any SQS
// message to the worker
const sqsBatchSize = 200

// Server contains a router and client interfaces to downstream services.
type Server struct {
	mux               *goji.Mux
	dynamoDB          dynamo.Dynamo
	redshift          redshift.Redshift
	dirtyTahoeReplica redshift.Redshift
	tahoeReplica      redshift.Redshift
	sqs               sqs.SQS
	statsd            stats.StatsdClient
	sqsstatsd         stats.StatsdClient
	twitchcon         twitchcon.TicketPurchaser
	dbWriter          db.Writer
}

// ServerParams contains the required components for instantiating a Server.
type ServerParams struct {
	DynamoDB          dynamo.Dynamo
	Redshift          redshift.Redshift
	DirtyTahoeReplica redshift.Redshift
	TahoeReplica      redshift.Redshift
	SQS               sqs.SQS
	Statsd            stats.StatsdClient
	SQSStatsd         stats.StatsdClient
	Twitchcon         twitchcon.TicketPurchaser
	DBWriter          db.Writer
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:               goji.NewMux(),
		sqs:               params.SQS,
		dynamoDB:          params.DynamoDB,
		redshift:          params.Redshift,
		dirtyTahoeReplica: params.DirtyTahoeReplica,
		tahoeReplica:      params.TahoeReplica,
		statsd:            params.Statsd,
		sqsstatsd:         params.SQSStatsd,
		twitchcon:         params.Twitchcon,
		dbWriter:          params.DBWriter,
	}

	server.mux.Use(httputil.PanicRecoverer)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(server.statsd.StatsLogger)

	root.HandleFunc(pat.Post("/process"), server.process)

	cron := goji.SubMux()
	root.Handle(pat.New("/cron/*"), cron)
	cron.HandleFunc(pat.Post("/n_broadcast_days_30_days"), server.nBroadcastDays30Days)
	cron.HandleFunc(pat.Post("/n_concurrents_30_days"), server.nConcurrents30Days)
	cron.HandleFunc(pat.Post("/n_days_no_viewers"), server.nDaysNoViewers)
	cron.HandleFunc(pat.Post("/n_hours_30days"), server.nHours30Days)
	cron.HandleFunc(pat.Post("/n_hours_lifetime"), server.nHoursLifetime)
	cron.HandleFunc(pat.Post("/n_minute_watched_lifetime"), server.nMinuteWatchedLifetime)
	cron.HandleFunc(pat.Post("/n_raid_consecutive_broadcast"), server.nRaidConsecutiveBroadcast)
	cron.HandleFunc(pat.Post("/n_unique_chatter_broadcast"), server.nUniqueChatterBroadcast)
	cron.HandleFunc(pat.Post("/n_viewers_lifetime"), server.nViewers)
	cron.HandleFunc(pat.Post("/n_followers"), server.nFollowers)
	cron.HandleFunc(pat.Post("/twitchcon"), server.handleTwitchcon)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}

// SaveSQSState saves message counts ot statsd whenver we create a batch of events
// so we can monitor the state of the queue
func (s *Server) SaveSQSState(achievement string, size int64) {
	// I don't have a lot of experience with statsd and I don't understand why
	// these 2 calls are being made this way. Why didn't they call Inc once with
	// size?
	err := s.sqsstatsd.Inc(fmt.Sprintf("sqs.%s.messages", achievement), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = s.sqsstatsd.Inc(fmt.Sprintf("sqs.%s.messages", achievement), int64(size)-1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = s.sqsstatsd.Inc(fmt.Sprintf("sqs.%s.sent", achievement), int64(size), 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}
