package sourcer

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"time"

	"github.com/pkg/errors"

	"code.justin.tv/cb/achievements/internal/achievement/singlebroadcasttitle"
	"code.justin.tv/cb/achievements/internal/achievement/singleselectgamecommunity"
	log "github.com/sirupsen/logrus"
)

type channelUpdateInputMessage struct {
	Changes channelUpdateObject `json:"changes"`
}

type channelUpdateObject struct {
	ChannelID int64   `json:"id"`
	Game      *string `json:"game"`
	Status    *string `json:"status"`
}

func (s *Server) processChannelUpdateEvent(ctx context.Context, input string) error {
	var message channelUpdateInputMessage

	err := json.Unmarshal([]byte(input), &message)
	if err != nil {
		return errors.Wrap(err, "error: failed to parse channel_update_event")
	}

	// discard message if channel id is not valid
	if message.Changes.ChannelID == 0 {
		log.WithField("input", input).Info("processChannelUpdateEvent() discard message with ChannelID == 0")
		return nil
	}

	// if neither game or status were changed, then we ignore this message
	if message.Changes.Game == nil && message.Changes.Status == nil {
		return nil
	}

	channelID := strconv.FormatInt(message.Changes.ChannelID, 10)

	// check if the game or status has changed before
	channelUpdate, err := s.dynamoDB.GetChannelUpdate(ctx, channelID)
	if err != nil {
		return errors.Wrap(err, "error: failed to GetChannelUpdate() while processChannUpdateEvent()")
	}

	isGameCommunityUpdated := channelUpdate.GameUpdatedAt != nil || channelUpdate.CommunityUpdatedAt != nil
	isTitleUpdated := channelUpdate.TitleUpdatedAt != nil

	// both first_game_community_update and first_broadcast_title achievements
	// have been recorded in dynamo, nothing to do
	if isGameCommunityUpdated && isTitleUpdated {
		return nil
	}

	now := time.Now().UTC()

	if !isGameCommunityUpdated && message.Changes.Game != nil && *message.Changes.Game != "" {
		channelUpdate.GameUpdatedAt = &now

		err = s.sqs.Send(ctx, singleselectgamecommunity.Key, &singleselectgamecommunity.Input{
			ChannelID: channelID,
		})

		go s.SaveSQSState(singleselectgamecommunity.Key, 1)

		if err != nil {
			return errors.Wrap(err, fmt.Sprintf("error: failed to send %s to worker", singleselectgamecommunity.Key))
		}
	}

	if !isTitleUpdated && message.Changes.Status != nil && *message.Changes.Status != "" {
		channelUpdate.TitleUpdatedAt = &now

		err = s.sqs.Send(ctx, singlebroadcasttitle.Key, &singlebroadcasttitle.Input{
			ChannelID: channelID,
		})

		go s.SaveSQSState(singlebroadcasttitle.Key, 1)

		if err != nil {
			return errors.Wrap(err, fmt.Sprintf("error: failed to send %s to worker sqs", singlebroadcasttitle.Key))
		}
	}

	go func() {
		err := s.dynamoDB.SaveChannelUpdate(context.Background(), channelUpdate)
		if err != nil {
			log.WithError(err).Error("error: failed to cache channel_update")
		}
	}()

	return nil
}
