package worker

import (
	"context"

	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/sns"
	log "github.com/sirupsen/logrus"
)

func (s *Server) sendAchievementNotifications(key string, completedProgressions []db.Progression) {
	featureEnabled, err := s.dynamoDB.IsFeatureEnabled(context.Background(), dynamo.SendAchievementSNSFeature)
	if err != nil {
		log.WithError(err).Error("failed to read featuredEnabled flag")
		return
	}

	if !featureEnabled || len(completedProgressions) == 0 {
		return
	}

	for _, completedProgression := range completedProgressions {
		go func(progression db.Progression) {
			message := sns.NotificationMessage{
				Key:         key,
				ChannelID:   progression.ChannelID,
				CompletedAt: *progression.CompletedAtUTC,
			}

			err := s.sns.PublishNotification(context.Background(), message)
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"achievement_key": key,
					"channel":         progression.ChannelID,
				}).Error("worker: failed to notify achievement completion")
			}

			err = s.notifier.PublishAchievementComplete(context.Background(), key, progression.ChannelID, progression.AchievementLevel)
			// We don't need to log errors if notifier publishing fails. Onsite notifications aren't
			// critical so a warning is fine.
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"achievement_key": key,
					"channel":         progression.ChannelID,
				}).Warn("worker: failed to send message to notifier")
			}
		}(completedProgression)
	}

	return
}
