package worker

import (
	"io/ioutil"
	"net/http"
	"time"

	"code.justin.tv/cb/achievements/internal/achievement"
	"code.justin.tv/cb/achievements/internal/clients/sqs"
	log "github.com/sirupsen/logrus"
)

// reqHeaderAchievementKey is the required request header
// for the Server.process handler ("POST /process") to determine
// the type of Achievement to process.
const reqHeaderAchievementKey = "X-Aws-Sqsd-Attr-" + sqs.MessageAttributeKeyAchievementKey

// process is the handler for the /process endpoint which will handle
// incoming sqs events
func (s *Server) process(w http.ResponseWriter, req *http.Request) {
	if s.dbDisabled {
		w.WriteHeader(http.StatusTeapot)
		return
	}

	achievementKey := req.Header.Get(reqHeaderAchievementKey)

	if len(achievementKey) == 0 {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	body, err := ioutil.ReadAll(req.Body)
	if err != nil {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	clients := &achievement.Clients{
		DBReader: s.dbReader,
		DBWriter: s.dbWriter,
	}

	job := s.newJob(achievementKey, clients, body)
	if job == nil {
		log.WithFields(log.Fields{
			"achievement_key": achievementKey,
			"request_body":    string(body),
		}).Error("app: key provided from worker queue did not match any achievement")

		w.WriteHeader(http.StatusBadRequest)
		return
	}

	jobStart := time.Now()

	completedJob, err := job.Process(req.Context())
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"achievement_key": achievementKey,
			"request_body":    string(body),
		}).Error("app: failed to process event")

		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	// send tracking for each completed progression
	log.Infof("app: %s updated", achievementKey)

	s.notifyJobCompletion(req.Context(), achievementKey, completedJob, jobStart)

	w.WriteHeader(http.StatusOK)
}
