package worker

import (
	"net/http"

	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/sns"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/clients/tracking"
	"code.justin.tv/cb/achievements/internal/httputil"
	"code.justin.tv/cb/achievements/internal/notifications"

	goji "goji.io"
	"goji.io/pat"
)

// Server contains a router and client interfaces to downstream services.
type Server struct {
	mux        *goji.Mux
	dbDisabled bool
	dbReader   db.Reader
	dbWriter   db.Writer
	dynamoDB   dynamo.Dynamo
	statsd     stats.StatsdClient
	sqsstatsd  stats.StatsdClient
	sns        sns.SNS
	notifier   notifications.Notifier
	tracking   tracking.Tracking
}

// ServerParams contains the required components of a Server.
type ServerParams struct {
	DBDisabled bool
	DBReader   db.Reader
	DBWriter   db.Writer
	Statsd     stats.StatsdClient
	SQSStatsd  stats.StatsdClient
	SNS        sns.SNS
	DynamoDB   dynamo.Dynamo
	Notifier   notifications.Notifier
	Tracking   tracking.Tracking
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:        goji.NewMux(),
		dbDisabled: params.DBDisabled,
		dbReader:   params.DBReader,
		dbWriter:   params.DBWriter,
		statsd:     params.Statsd,
		sqsstatsd:  params.SQSStatsd,
		sns:        params.SNS,
		dynamoDB:   params.DynamoDB,
		notifier:   params.Notifier,
		tracking:   params.Tracking,
	}

	server.mux.Use(httputil.PanicRecoverer)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.AccessLogger)
	root.Use(server.statsd.StatsLogger)

	root.HandleFunc(pat.Post("/process"), server.process)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
