package worker

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/cb/achievements/internal/achievement"
	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	log "github.com/sirupsen/logrus"
)

func (s *Server) notifyJobCompletion(ctx context.Context, achievementKey string, completedJob *achievement.CompletedJob, jobStart time.Time) {
	go s.timeJobCompletion(achievementKey, jobStart)
	go s.countAchievementProgress(achievementKey, completedJob)

	updatedCompleted := []db.Progression{}
	for _, update := range completedJob.UpdatedProgressions {
		if update.IsComplete() {
			updatedCompleted = append(updatedCompleted, update)
		}
	}

	trackingEnabled, err := s.dynamoDB.IsFeatureEnabled(ctx, dynamo.SpadeTrackingFeature)
	if err == nil && trackingEnabled {
		go s.tracking.TrackCompletedAchievements(updatedCompleted)
	}

	go s.sendAchievementNotifications(achievementKey, updatedCompleted)

	go func() {
		err := s.sqsstatsd.Dec(fmt.Sprintf("sqs.%s.messages", achievementKey), 1, 1)
		if err != nil {
			log.WithError(err).Error("failed to send stat to statsd")
		}
	}()
}

func (s *Server) timeJobCompletion(achievementKey string, jobStart time.Time) {
	err := s.statsd.TimingDuration(fmt.Sprintf("job.success.%s", achievementKey), time.Since(jobStart), 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

func (s *Server) countAchievementProgress(achievementKey string, completedJob *achievement.CompletedJob) {
	for _, progression := range completedJob.CompletedProgressions {
		err := s.statsd.Inc(fmt.Sprintf("job.%s.progress", achievementKey), int64(progression.Progress), 1)
		if err != nil {
			log.WithError(err).Error("failed to send stat to statsd")
		}
	}
}
