# Elastic Beanstalk App
resource "aws_iam_role" "app" {
  name = "${var.name}-${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_instance_profile" "app" {
  name = aws_iam_role.app.name
  role = aws_iam_role.app.name
}

resource "aws_iam_role_policy" "worker_tier" {
  name = "worker-tier-policy"
  role = aws_iam_role.app.name

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "MetricsAccess",
      "Action": [
        "cloudwatch:PutMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "XRayAccess",
      "Action":[
        "xray:PutTraceSegments",
        "xray:PutTelemetryRecords"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "QueueAccess",
      "Action": [
        "sqs:ChangeMessageVisibility",
        "sqs:DeleteMessage",
        "sqs:ReceiveMessage",
        "sqs:SendMessage"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "BucketAccess",
      "Action": [
        "s3:Get*",
        "s3:List*",
        "s3:PutObject"
      ],
      "Effect": "Allow",
      "Resource": [
        "arn:aws:s3:::elasticbeanstalk-*",
        "arn:aws:s3:::elasticbeanstalk-*/*"
      ]
    },
    {
      "Sid": "DynamoPeriodicTasks",
      "Action": [
        "dynamodb:BatchGetItem",
        "dynamodb:BatchWriteItem",
        "dynamodb:DeleteItem",
        "dynamodb:GetItem",
        "dynamodb:PutItem",
        "dynamodb:Query",
        "dynamodb:Scan",
        "dynamodb:UpdateItem"
      ],
      "Effect": "Allow",
      "Resource": [
        "arn:aws:dynamodb:*:*:table/*-stack-AWSEBWorkerCronLeaderRegistry*"
      ]
    },
    {
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:GetLogEvents",
        "logs:PutLogEvents",
        "logs:DescribeLogGroups",
        "logs:DescribeLogStreams",
        "logs:PutRetentionPolicy"
      ],
      "Resource": [
        "arn:aws:logs:us-west-2:*:*",
        "arn:aws:logs:*:*:log-group:/aws/elasticbeanstalk*"
      ]
    },
    {
      "Effect": "Allow",
      "Action": [
          "secretsmanager:GetResourcePolicy",
          "secretsmanager:GetSecretValue",
          "secretsmanager:DescribeSecret",
          "secretsmanager:ListSecretVersionIds",
          "secretsmanager:ListSecrets"
      ],
      "Resource": [
          "*"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "sandstorm" {
  name = "sandstorm-policy"
  role = aws_iam_role.app.name

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Resource": "${var.sandstorm_iam_role_arn}"
    },
    {
      "Action": "s3:GetObject",
      "Effect": "Allow",
      "Resource": "arn:aws:s3:::twitch-sandstorm/sandstorm-agent.rpm",
      "Sid": "Stmt1473817796000"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "dynamodb" {
  name = "dynamodb-policy"
  role = aws_iam_role.app.id

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "dynamodb:*",
      "Effect": "Allow",
      "Resource": [
        "${aws_dynamodb_table.first_streams.arn}",
        "${aws_dynamodb_table.single-dashboard-visit.arn}",
        "${aws_dynamodb_table.channel_updates.arn}",
        "${aws_dynamodb_table.feature_flags.arn}"
      ]
    },
    {
      "Action": "dynamodb:Query",
      "Effect": "Allow",
      "Resource": [
        "${aws_dynamodb_table.first_streams.arn}/index/*",
        "${aws_dynamodb_table.single-dashboard-visit.arn}/index/*",
        "${aws_dynamodb_table.channel_updates.arn}/index/*",
        "${aws_dynamodb_table.feature_flags.arn}/index/*"
      ]
    },
    {
      "Effect": "Allow",
      "Action": [
        "dynamodb:UpdateGlobalTable",
        "dynamodb:UpdateGlobalTableSettings"
      ],
      "Resource": "*"
    }
  ]
}
POLICY
}

# Redshift
resource "aws_iam_role" "redshift" {
  name = "${var.name}-${var.environment}-redshift"

  assume_role_policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        "Service": ["redshift.amazonaws.com", "ec2.amazonaws.com"]
      },
      "Sid": ""
    }
  ]
}
POLICY
}

resource "aws_iam_role_policy" "redshift" {
  name = "redshift-policy"
  role = aws_iam_role.redshift.name

  policy = <<POLICY
{
"Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:GetBucketAcl"
      ],
      "Resource": [
        "arn:aws:s3:::${aws_s3_bucket.redshift_logs.bucket}"
      ]
    },
    {
      "Effect": "Allow",
      "Action": [
        "s3:PutObject"
      ],
      "Resource": [
        "arn:aws:s3:::${aws_s3_bucket.redshift_logs.bucket}",
        "arn:aws:s3:::${aws_s3_bucket.redshift_logs.bucket}/*"
      ]
    },
    {
      "Effect": "Allow",
      "Action": "redshift:DescribeClusters",
      "Resource": "*"
    },
    {
      "Effect": "Allow",
      "Action": [
        "redshift:ModifyClusterIamRoles",
        "redshift:CreateCluster"
      ],
      "Resource": [
        "arn:aws:redshift:${var.aws_region}:${var.account_id}:cluster:${aws_redshift_cluster.default.cluster_identifier}"
      ]
    },
    {
      "Effect": "Allow",
      "Action": "iam:PassRole",
      "Resource": [
        "arn:aws:iam::${var.account_id}:role/redshift"
      ]
    }
  ]
}
POLICY
}

resource "aws_iam_role" "firehose" {
  name = "spade-downstream-prod-${var.name}-${var.environment}-firehose"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "firehose.amazonaws.com"
      },
      "Effect": "Allow",
      "Condition": {
        "StringEquals": {
          "sts:ExternalId":"${var.account_id}"
        }
      }
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "firehose" {
  name = "spade-downstream-prod-${var.name}-${var.environment}-firehose"
  role = aws_iam_role.firehose.id

  policy = <<EOF
{
"Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:AbortMultipartUpload",
        "s3:GetBucketLocation",
        "s3:GetObject",
        "s3:ListBucket",
        "s3:ListBucketMultipartUploads",
        "s3:PutObject"
      ],
      "Resource": [
        "arn:aws:s3:::${aws_s3_bucket.data_sink.bucket}",
        "arn:aws:s3:::${aws_s3_bucket.data_sink.bucket}/*"
      ]
    },
    {
      "Sid": "PushToCloudwatchLogs",
      "Effect": "Allow",
      "Action": [
          "logs:CreateLogGroup",
          "logs:CreateLogStream",
          "logs:PutLogEvents"
      ],
      "Resource": [
          "arn:aws:logs:*:*:*"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role" "spade_putter" {
  name = "spade-downstream-prod-${var.name}-${var.environment}-putter"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "AWS": [
          "arn:aws:iam::673385534282:root"
        ]
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "spade_putter" {
  name = "spade-downstream-prod-${var.name}-${var.environment}-putter"
  role = aws_iam_role.spade_putter.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "firehose:PutRecordBatch"
      ],
      "Effect": "Allow",
      "Resource": [
        "arn:aws:firehose:${var.aws_region}:${var.account_id}:deliverystream/spade-downstream-prod-cb-ach-${var.abbreviated_environment_name}*"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "eventbus_access_policy" {
  role       = aws_iam_role.app.name
  policy_arn = data.aws_cloudformation_stack.eventbus.outputs["EventBusAccessPolicyARN"]
}
