resource "aws_redshift_subnet_group" "achievements" {
  name        = "${var.name}-${var.environment}-redshift-subnet"
  description = "CB Achievements redshift subnet group"
  # Public subnets comes back from the AWS API in a seemingly random order
  # and this prevents unnecessary changes to Beanstalk environments.
  subnet_ids  = split(",", data.terraform_remote_state.account.outputs.public_subnets)

  tags = {
    environment = var.environment
    Name        = var.name
    Owner       = var.owner
    Project     = var.project
    Team        = var.team
  }
}

resource "aws_redshift_cluster" "default" {
  allow_version_upgrade               = "false"
  availability_zone                   = "us-west-2a"
  automated_snapshot_retention_period = var.redshift_automated_snapshot_retention_period
  cluster_identifier                  = var.redshift_cluster_identifier
  cluster_parameter_group_name        = aws_redshift_parameter_group.achievements-param-group.name
  cluster_subnet_group_name           = aws_redshift_subnet_group.achievements.name
  cluster_type                        = "multi-node"
  cluster_version                     = "1.0"
  database_name                       = var.redshift_database_name
  encrypted                           = "false"
  enhanced_vpc_routing                = "false"
  iam_roles                           = [aws_iam_role.redshift.arn]
  preferred_maintenance_window        = "wed:20:00-wed:20:30"

  logging {
    enable        = "true"
    bucket_name   = aws_s3_bucket.redshift_logs.bucket
    s3_key_prefix = "redshift_logs/"
  }

  master_username        = var.redshift_master_username
  master_password        = var.redshift_temporary_password
  node_type              = "dc2.8xlarge"
  number_of_nodes        = "8"
  port                   = "5439"
  publicly_accessible    = "true"
  skip_final_snapshot    = "true"
  vpc_security_group_ids = [data.terraform_remote_state.account.outputs.twitch_subnets_sg]
}

resource "aws_redshift_parameter_group" "achievements-param-group" {
  name   = "achievements-${var.environment}-v1"
  family = "redshift-1.0"

  parameter {
    name  = "wlm_json_configuration"
    value = "[{\"query_concurrency\":10}]"
  }
}
