# Cloudwatch alarms

resource "aws_cloudwatch_metric_alarm" "stream-too-few-bytes" {
  count               = length(var.redshift_tables) * var.alarm_on_low_number_of_events
  alarm_name          = "spade-downstream-prod-${var.team_name}-${element(var.redshift_tables, count.index)}-too-few-bytes"
  alarm_description   = "Fewer than .1% of max bytes sent to spade-downstream-prod-${var.team_name}-${element(var.redshift_tables, count.index)}"
  comparison_operator = "LessThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = 0.001 * var.capacity_increase_multiplier * 5000000 * 60
  treat_missing_data  = "breaching"

  dimensions = {
    DeliveryStreamName = element(aws_kinesis_firehose_delivery_stream.stream.*.name, count.index)
  }

  alarm_actions             = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
  ok_actions                = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
  insufficient_data_actions = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
}

resource "aws_cloudwatch_metric_alarm" "stream-bytes" {
  count               = length(var.redshift_tables)
  alarm_name          = "spade-downstream-prod-${var.team_name}-${var.environment}-${element(var.redshift_tables, count.index)}-too-many-bytes"
  alarm_description   = "Too many bytes sent to spade-downstream-prod-${var.team_name}-${var.environment}-${element(var.redshift_tables, count.index)}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = 0.85 * var.capacity_increase_multiplier * 5000000 * 60
  treat_missing_data  = "notBreaching"

  dimensions = {
    DeliveryStreamName = element(aws_kinesis_firehose_delivery_stream.stream.*.name, count.index)
  }

  alarm_actions             = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
  ok_actions                = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
  insufficient_data_actions = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
}

resource "aws_cloudwatch_metric_alarm" "stream-records" {
  count               = length(var.redshift_tables)
  alarm_name          = "spade-downstream-prod-${var.team_name}-${var.environment}-${element(var.redshift_tables, count.index)}-too-many-records"
  alarm_description   = "Too many records sent to spade-downstream-prod-${var.team_name}-${var.environment}-${element(var.redshift_tables, count.index)}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingRecords"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = 0.85 * var.capacity_increase_multiplier * 5000 * 60
  treat_missing_data  = "notBreaching"

  dimensions = {
    DeliveryStreamName = element(aws_kinesis_firehose_delivery_stream.stream.*.name, count.index)
  }

  alarm_actions             = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
  ok_actions                = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
  insufficient_data_actions = [var.scieng_sns_topic, aws_sns_topic.firehose_capacity.arn]
}

resource "aws_cloudwatch_log_group" "firehose-log-group" {
  count = length(var.redshift_tables)
  name  = "cb-achievements-${var.environment}-firehose-${element(var.redshift_tables, count.index)}"

  tags = {
    Environment = var.environment
  }
}

resource "aws_cloudwatch_log_stream" "firehose-log-stream-redshift" {
  count          = length(var.redshift_tables)
  name           = "redshift"
  log_group_name = aws_cloudwatch_log_group.firehose-log-group[count.index].name
}

resource "aws_cloudwatch_log_stream" "firehose-log-stream-s3" {
  count          = length(var.redshift_tables)
  name           = "s3"
  log_group_name = aws_cloudwatch_log_group.firehose-log-group[count.index].name
}
