# Variables for configuring firehose streams
# From the scieng twitch docs on setting up a kinesis stream:
#
# https://git.xarth.tv/twitch/docs/blob/2dcd0bf545fdb6645988d2b1f2243adb6756a6ec/scieng/kinesis/README.md
#
# - `bucket_name`: The S3 bucket the stream will write to. It will be
#     created
# unless you delete the aws_s3_bucket resource.
# - `bucket-prefix`: Prefix to write data into (including a trailing slash
#     unless it's empty).
# - `expiration-days`: Number of days to keep data in the receiving s3
#     bucket.
# - `account-id`: Your AWS account ID. Available with `aws sts
#     get-caller-identity`.
# - `data.null_data_source.resource_names.inputs.sns_topic`: An SNS topic in your account that will receive alerts
#     when the stream
# nears write saturation, so you will know to increase the stream capacity.
# - `capacity_increase_multiplier`: Multiplier of how much you increased
#     the stream capacity (see below).
# Used to trigger an alarm on nearing stream capacity. If you doubled
#     stream capacity, this would be 2.
# - `jdbcurl` (Redshift only): JDBC URL to the target Redshift cluster. It
#     can be found on the Redshift console.
# - `redshift-username` (Redshift only): Username to use to connect to the
#     Redshift cluster.
# - `redshift-password` (Redshift only): Password for above user.
# - `firehose-names` (Redshift only): Names of the tables you'll be writing
#     into. A Firehose will be created for each output table (you can have
#     multiple events written into one table).
# - `alarm_on_low_number_of_events`: This should be either `1` (opt in) or
#     `0` (opt out)
#   if you want to alarm (you+scieng) when your stream has <1% of the full
#     capacity coming
#   through. This is primarily to catch instances where there are either
#     problems with spade
#     sending data, or with the source data failing to send data.

variable "event_name" {
  type        = string
  description = "The name of the invent being ingested (eg. minute_broadcast)"
}

variable "role" {
  type = string
}

variable "environment" {
  type = string
}

data "null_data_source" "resource_names" {
  inputs = {
    bucket_prefix = var.event_name
  }
}

variable "bucket_name" {
  description = "Name of S3 bucket to dump data into"
}

variable "team_name" {
  type        = string
  default     = "cb-ach"
  description = "Name of your team. Used for namespacing in cloudwatch alarms to avoid collisions"
}

variable "account_id" {
  default     = "989470033077"
  description = "Your AWS account ID."
}

variable "redshift_username" {
  type = string
}

variable "redshift_password" {
  type = string
}

variable "redshift_jdbc_url" {
  type = string
}

# Names of the tables you'll be writing into.
# A Firehose will be created for each output table
# (you can have multiple events written into one table).
variable "redshift_tables" {
  type = list
}

# Firehose capacity can only be increased by a multiplier and with a
# support ticket. If you increased your capacity, fill in the multiplier
# you increased your capacity by to get useful alarms. E.g. if you doubled
# your Firehose capacity, the value here is 2.
variable "capacity_increase_multiplier" {
  default = 2
}

variable "scieng_sns_topic" {
  default = "arn:aws:sns:us-west-2:673385534282:spade-downstream-prod-alert"
}

# This should be either `1` (opt in) or `0` (opt out)
# if you want to alarm (you+scieng) when your stream has <1% of the full capacity coming
# through. This is primarily to catch instances where there are either problems with spade
# sending data, or with the source data failing to send data.
variable "alarm_on_low_number_of_events" {
  default = 1
}
