# Worker queues
resource "aws_sqs_queue" "worker_queue" {
  name = "${var.name}-${var.environment}-worker-queue"

  visibility_timeout_seconds = var.sqs_defaults["visibility_timeout_seconds"]
  message_retention_seconds  = 345600 # 4 days

  redrive_policy = <<EOF
{
  "maxReceiveCount":5,
  "deadLetterTargetArn": "${aws_sqs_queue.worker_queue_dlq.arn}"
}
EOF

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_sqs_queue" "worker_queue_dlq" {
  name                      = "${var.name}-${var.environment}-worker-queue-dlq"
  message_retention_seconds = var.sqs_defaults["dead_message_retention_seconds"]

  lifecycle {
    prevent_destroy = true
  }
}

# Quests queue
resource "aws_sqs_queue" "quests_queue" {
  name = "${var.name}-${var.environment}-quests-queue"

  visibility_timeout_seconds = var.sqs_defaults["visibility_timeout_seconds"]
  message_retention_seconds  = var.sqs_defaults["message_retention_seconds"]

  redrive_policy = <<EOF
{
  "maxReceiveCount":1,
  "deadLetterTargetArn": "${aws_sqs_queue.quests_queue_dlq.arn}"
}
EOF

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_sqs_queue_policy" "quests_queue_permissions" {
  queue_url = aws_sqs_queue.quests_queue.id

  # Define SNS topics that may publish to this SQS queue
  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "arn:aws:sqs:${var.aws_region}:${var.account_id}:${var.name}-${var.environment}-quests-queue/SNSPolicy",
  "Statement": [
    {
      "Sid": "Stmt1473987189364",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "arn:aws:sqs:${var.aws_region}:${var.account_id}:${var.name}-${var.environment}-quests-queue",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": [
            "${aws_sns_topic.stream-manager-visited.arn}",
            "${var.sns_topic_users_service_channel_mutation}",
            "${var.sns_topic_zuma_channel_select_community}",
            "${aws_sns_topic.notify-video-ingest-stream.arn}"
          ]
        }
      }
    }
  ]
}
POLICY
}

resource "aws_sqs_queue" "quests_queue_dlq" {
  name                      = "${var.name}-${var.environment}-quests-queue-dlq"
  message_retention_seconds = var.sqs_defaults["dead_message_retention_seconds"]

  lifecycle {
    prevent_destroy = true
  }
}

# Affiliates queue
resource "aws_sqs_queue" "affiliates_queue" {
  name = "${var.name}-${var.environment}-affiliates-queue"

  visibility_timeout_seconds = var.sqs_defaults["visibility_timeout_seconds"]
  message_retention_seconds  = var.sqs_defaults["message_retention_seconds"]

  redrive_policy = <<EOF
{
  "maxReceiveCount":1,
  "deadLetterTargetArn": "${aws_sqs_queue.affiliates_queue_dlq.arn}"
}
EOF

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_sqs_queue" "affiliates_queue_dlq" {
  name                      = "${var.name}-${var.environment}-affiliates-queue-dlq"
  message_retention_seconds = var.sqs_defaults["dead_message_retention_seconds"]

  lifecycle {
    prevent_destroy = true
  }
}

## Quests queue sns subscriptions for It Begins
resource "aws_sns_topic_subscription" "stream_manager_visited_quests" {
  topic_arn = aws_sns_topic.stream-manager-visited.arn
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.quests_queue.arn
}

resource "aws_sns_topic_subscription" "channel_mutation_quests" {
  topic_arn = var.sns_topic_users_service_channel_mutation
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.quests_queue.arn
}

resource "aws_sns_topic_subscription" "channel_select_community_quests" {
  topic_arn = var.sns_topic_zuma_channel_select_community
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.quests_queue.arn
}

resource "aws_sns_topic_subscription" "first_tream_quests" {
  topic_arn = aws_sns_topic.notify-video-ingest-stream.arn
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.quests_queue.arn
}

## Sourcer Queues
resource "aws_sqs_queue" "sourcer_queue" {
  name = "${var.name}-${var.environment}-sourcer-queue"

  visibility_timeout_seconds = var.sqs_defaults["visibility_timeout_seconds"]
  message_retention_seconds  = var.sqs_defaults["message_retention_seconds"]

  redrive_policy = <<EOF
{
  "maxReceiveCount":1,
  "deadLetterTargetArn": "${aws_sqs_queue.sourcer_queue_dlq.arn}"
}
EOF

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_sqs_queue_policy" "sourcer_queue_permissions" {
  queue_url = aws_sqs_queue.sourcer_queue.id

  # Define SNS topics that may publish to this SQS queue
  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "arn:aws:sqs:${var.aws_region}:${var.account_id}:${var.name}-${var.environment}-sourcer-queue/SNSPolicy",
  "Statement": [
    {
      "Sid": "Stmt1473987189364",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "arn:aws:sqs:${var.aws_region}:${var.account_id}:${var.name}-${var.environment}-sourcer-queue",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": [
            "${aws_sns_topic.stream-manager-visited.arn}",
            "${var.sns_topic_users_service_channel_mutation}",
            "${var.sns_topic_autohost_list_updates}",
            "${var.sns_topic_zuma_channel_select_community}",
            "${aws_sns_topic.notify-video-ingest-stream.arn}"
          ]
        }
      }
    }
  ]
}
POLICY
}

resource "aws_sqs_queue" "sourcer_queue_dlq" {
  name                      = "${var.name}-${var.environment}-sourcer-queue-dlq"
  message_retention_seconds = var.sqs_defaults["dead_message_retention_seconds"]

  lifecycle {
    prevent_destroy = true
  }
}

# Various SNS subscriptions for sourcer_queue
resource "aws_sns_topic_subscription" "stream_manager_visited_sourcer" {
  topic_arn = aws_sns_topic.stream-manager-visited.arn
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.sourcer_queue.arn
}

resource "aws_sns_topic_subscription" "users_service_channel_mutation" {
  topic_arn = var.sns_topic_users_service_channel_mutation
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.sourcer_queue.arn
}

resource "aws_sns_topic_subscription" "autohost_list_updates" {
  topic_arn = var.sns_topic_autohost_list_updates
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.sourcer_queue.arn
}

resource "aws_sns_topic_subscription" "zuma_channel_select_community" {
  topic_arn = var.sns_topic_zuma_channel_select_community
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.sourcer_queue.arn
}

resource "aws_sns_topic_subscription" "video_ingest_stream_notifications" {
  topic_arn = aws_sns_topic.notify-video-ingest-stream.arn
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.sourcer_queue.arn
}

resource "aws_sns_topic_subscription" "affiliates_onboarding_status" {
  topic_arn = var.sns_topic_affiliates_onboarding_status
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.affiliates_queue.arn
}

resource "aws_sqs_queue_policy" "worker_queue_permissions" {
  queue_url = aws_sqs_queue.affiliates_queue.id

  # Define SNS topics that may publish to this SQS queue
  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "${aws_sqs_queue.affiliates_queue.arn}/SQSDefaultPolicy",
  "Statement": [
    {
      "Sid": "Sid1533672434115",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.affiliates_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": [
            "${var.sns_topic_affiliates_onboarding_status}"
          ]
        }
      }
    }
  ]
}
POLICY
}

# Eventbus queue
resource "aws_sqs_queue" "eventbus_queue_dlq" {
  name              = "${var.name}-${var.environment}-eventbus-queue-dlq"
  kms_master_key_id = data.aws_cloudformation_stack.eventbus.outputs["EventBusKMSMasterKeyARN"]
}

resource "aws_sqs_queue" "eventbus_queue" {
  name              = "${var.name}-${var.environment}-eventbus-queue"
  policy            = data.aws_cloudformation_stack.eventbus.outputs["EventBusSQSPolicyDocument"]
  kms_master_key_id = data.aws_cloudformation_stack.eventbus.outputs["EventBusKMSMasterKeyARN"]

  visibility_timeout_seconds = var.sqs_defaults["visibility_timeout_seconds"]
  message_retention_seconds  = var.sqs_defaults["message_retention_seconds"]

  redrive_policy = <<EOF
{
  "maxReceiveCount":1,
  "deadLetterTargetArn": "${aws_sqs_queue.eventbus_queue_dlq.arn}"
}
EOF
}
