terraform {
  required_version = ">= 0.13.5"

  backend "s3" {
    bucket  = "cb-achievements-terraform"
    key     = "production/terraform.tfstate"
    region  = "us-west-2"
    profile = "twitch-cb-aws"
    encrypt = true
  }
}

data "aws_secretsmanager_secret_version" "slack_secrets_json" {
  secret_id = "achievements/production/slack"
}

locals {
  slack_secrets = jsondecode(
    data.aws_secretsmanager_secret_version.slack_secrets_json.secret_string
  )
}

module "achievements" {
  source = "../modules/achievements"

  environment = "production"

  aws_key_name        = "cb/achievements-production"
  eb_api_asg_max_size = 20
  eb_api_asg_min_size = 10

  postgres_master_identifier  = "cb-achievements-production-5"
  postgres_replica_identifier = "cb-achievements-production-replica-5"
  postgres_parameter_group    = "cb-achievements-production-postgres10"

  aurora_cluster_identifier = "cb-achievements-production-aurora"

  sandstorm_iam_role_arn = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/cb-achievements-production"

  # Redshift vars
  redshift_cluster_identifier                  = "cb-achievements-production"
  redshift_database_name                       = "achievements"
  redshift_automated_snapshot_retention_period = "1"
  redshift_jdbc_url                            = "jdbc:redshift://cb-achievements-production.c8uuzrcq1xro.us-west-2.redshift.amazonaws.com:5439/achievements"

  # Suffix for firehose stream name, shortened to avoid char limits
  abbreviated_environment_name = "prd"

  sns_topic_users_service_channel_mutation   = "arn:aws:sns:us-west-2:641044725657:channel_mutation_events"
  sns_topic_autohost_list_updates            = "arn:aws:sns:us-west-2:447680546588:autohost_production_list_updates"
  sns_topic_zuma_channel_select_community    = "arn:aws:sns:us-west-2:603200399373:zuma_production_channel_select_community"
  sns_topic_affiliates_onboarding_status     = "arn:aws:sns:us-west-2:316550374861:OnboardingStatus"

  vpc_endpoint_service_allowed_principals = [
    "arn:aws:iam::903725031532:root", # twitch-tailwind-prod
    "arn:aws:iam::787149559823:root", # graphql-prod
    "arn:aws:iam::603200399373:root", # twitch-chat-aws
    "arn:aws:iam::239192441054:root", # cbt-martian-prod
  ]
}

resource "aws_sns_topic" "slack_alerts" {
  name = "cb-achievements-production-slack-alerts"
}

module "notify_slack" {
  source  = "terraform-aws-modules/notify-slack/aws"
  version = "~> 4.0"

  sns_topic_name       = aws_sns_topic.slack_alerts.name
  create_sns_topic     = false
  lambda_function_name = "cb-achievements-production-notify-slack"

  slack_webhook_url = local.slack_secrets.slack_webhook_alerts_url
  slack_channel     = "creator-analytics-alerts"
  slack_username    = "creator-analytics-alerts"

  depends_on = [aws_sns_topic.slack_alerts]
}

resource "aws_sns_topic" "pagerduty_alerts" {
  name = "cb-achievements-production-pagerduty-alerts"
}

resource "aws_sns_topic_subscription" "pagerduty_alerts_sns" {
  topic_arn              = aws_sns_topic.pagerduty_alerts.arn
  protocol               = "https"
  endpoint               = "https://events.pagerduty.com/integration/018251fed8c549cdb367ea0d8ddf39d1/enqueue"
  endpoint_auto_confirms = true
}

module "alarms" {
  source = "../modules/alarms"

  environment                  = "production"
  abbreviated_environment_name = "prd"
  alert_arns                   = [aws_sns_topic.pagerduty_alerts.arn, aws_sns_topic.slack_alerts.arn]
}

module "ripley" {
  source = "../modules/vpc_endpoint"

  vpc_id               = "vpc-7851b71e"
  subnet_ids           = ["subnet-6f4a7808", "subnet-45e7ad0c", "subnet-86fe23dd"]
  security_group_ids   = ["sg-57251d2c"]
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-0e9e8a2896b1d5e89"
  ssl_domain_name      = "prod.ripley.twitch.a2z.com"
}

module "bastion_host" {
  source = "../modules/bastionhost"

  subnet_ids         = ["subnet-6f4a7808", "subnet-45e7ad0c", "subnet-86fe23dd"]
  security_group_ids = ["sg-57251d2c"]
}

// Jarvis
resource "aws_iam_role" "twitch-telemetry-writer" {
  name = "twitch-telemetry-writer"

  assume_role_policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::578510050023:role/twitch-telemetry-writer-master"
      },
      "Sid": ""
    }
  ]
}
POLICY
}

resource "aws_iam_role_policy" "twitch-telemetry-writer" {
  name = "twitch-telemetry-writer-policy"
  role = aws_iam_role.twitch-telemetry-writer.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "cloudwatch:*"
      ],
      "Effect": "Allow",
      "Resource": [
        "*"
      ]
    }
  ]
}
EOF
}

module "dart_prod" {
  source = "../modules/vpc_endpoint"

  vpc_id               = "vpc-7851b71e"
  subnet_ids           = ["subnet-6f4a7808", "subnet-45e7ad0c", "subnet-86fe23dd"]
  security_group_ids   = ["sg-57251d2c"]
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-0ce6ec756b1a5acba"
  ssl_domain_name      = "us-west-2.prod.twitchdartreceiver.s.twitch.a2z.com"
}
