terraform {
  required_version = ">= 0.13.5"

  backend "s3" {
    bucket  = "cb-achievements-terraform"
    key     = "staging/terraform.tfstate"
    region  = "us-west-2"
    profile = "twitch-cb-aws"
    encrypt = true
  }
}

data "aws_secretsmanager_secret_version" "slack_secrets_json" {
  secret_id = "achievements/staging/slack"
}

locals {
  slack_secrets = jsondecode(
    data.aws_secretsmanager_secret_version.slack_secrets_json.secret_string
  )
}

module "achievements" {
  source = "../modules/achievements"

  environment = "staging"

  aws_key_name        = "cb/achievements-staging"
  eb_api_asg_max_size = 4
  eb_api_asg_min_size = 2

  postgres_master_identifier  = "cb-achievements-staging-2"
  postgres_replica_identifier = "cb-achievements-staging-replica-2"
  postgres_parameter_group    = "cb-achievements-staging-postgres10"

  aurora_cluster_identifier = "cb-achievements-staging-aurora"

  sandstorm_iam_role_arn = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/cb-achievements-staging"

  # Redshift Vars
  redshift_cluster_identifier                  = "cb-achievements-staging"
  redshift_database_name                       = null
  redshift_automated_snapshot_retention_period = "0" # Don't keep backups on staging
  redshift_jdbc_url                            = "jdbc:redshift://cb-achievements-staging.c8uuzrcq1xro.us-west-2.redshift.amazonaws.com:5439/dev"

  # Suffix for firehose stream name, shortened to avoid char limits
  abbreviated_environment_name = "stg"

  sns_topic_users_service_channel_mutation   = "arn:aws:sns:us-west-2:465369119046:channel_mutation_events"
  sns_topic_autohost_list_updates            = "arn:aws:sns:us-west-2:038504843107:autohost_staging_list_updates"
  sns_topic_zuma_channel_select_community    = "arn:aws:sns:us-west-2:603200399373:zuma_staging_channel_select_community"
  sns_topic_affiliates_onboarding_status     = "arn:aws:sns:us-west-2:589506732331:OnboardingStatus"

  vpc_endpoint_service_allowed_principals = [
    "arn:aws:iam::795258568209:root", # twitch-tailwind-dev
    "arn:aws:iam::645130450452:root", # graphql-dev
    "arn:aws:iam::603200399373:root", # twitch-chat-aws
    "arn:aws:iam::162224989036:root", # cbt-martian-dev
  ]
}

resource "aws_sns_topic" "slack_alerts" {
  name = "cb-achievements-staging-slack-alerts"
}

module "notify_slack" {
  source  = "terraform-aws-modules/notify-slack/aws"
  version = "~> 4.0"

  sns_topic_name       = aws_sns_topic.slack_alerts.name
  create_sns_topic     = false
  lambda_function_name = "cb-achievements-staging-notify-slack"

  slack_webhook_url = local.slack_secrets.slack_webhook_alerts_url
  slack_channel     = "creator-analytics-alerts"
  slack_username    = "creator-analytics-alerts"

  depends_on = [aws_sns_topic.slack_alerts]
}

module "alarms" {
  source = "../modules/alarms"

  environment                  = "staging"
  abbreviated_environment_name = "stg"
  alert_arns                   = [aws_sns_topic.slack_alerts.arn]
}

module "ripley" {
  source = "../modules/vpc_endpoint"

  vpc_id               = "vpc-7851b71e"
  subnet_ids           = ["subnet-6f4a7808", "subnet-45e7ad0c", "subnet-86fe23dd"]
  security_group_ids   = ["sg-57251d2c"]
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-0302ab0bc9133e1a4"
  ssl_domain_name      = "beta.ripley.twitch.a2z.com"
}

// Adapted from https://git.xarth.tv/chat/terraform/pull/2733/files#diff-9534f97ed0687f803b6d09ca125a0bfbR338-R357
module "dart_staging" {
  source = "../modules/vpc_endpoint"

  vpc_id               = "vpc-7851b71e"
  subnet_ids           = ["subnet-6f4a7808", "subnet-45e7ad0c", "subnet-86fe23dd"]
  security_group_ids   = ["sg-57251d2c"]
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-00cc5f5ddd42126a0"
  ssl_domain_name      = "us-west-2.beta.twitchdartreceiver.s.twitch.a2z.com"
}
