package lambda

import (
	"context"
	"encoding/json"
	"fmt"

	payload "code.justin.tv/cb/achievements/validator/model/lambda"
	"code.justin.tv/cb/achievements/validator/model/query"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	serviceLambda "github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/lambda/lambdaiface"
)

const (
	validatorLambdaARN = "arn:aws:lambda:us-west-2:989470033077:function:staging-Achievements-CleanTapValidator-Fn9270CBC0-QOoccZe2XT8y"
)

type Lambda interface {
	RunValidator(queryName query.Name) error
}

type lambda struct {
	client lambdaiface.LambdaAPI
}

func NewClient(sess *session.Session) Lambda {
	return &lambda{
		client: serviceLambda.New(sess),
	}
}

func (l *lambda) RunValidator(queryName query.Name) error {
	payload, err := json.Marshal(payload.Payload{
		QueryName: string(queryName),
	})
	if err != nil {
		return fmt.Errorf("marshal lambda payload with queryName %s: %w", queryName, err)
	}

	_, err = l.client.InvokeWithContext(context.Background(), &serviceLambda.InvokeInput{
		FunctionName: aws.String(validatorLambdaARN),
		Payload:      payload,
	})

	if err != nil {
		return fmt.Errorf("invoke validator lambda: %w", err)
	}

	return nil
}
