package main

import (
	"context"
	"os"
	"strconv"

	"code.justin.tv/cb/achievements/config"
	"code.justin.tv/cb/achievements/internal/clients/redshift"
	"code.justin.tv/cb/achievements/internal/clients/sandstorm"
	"code.justin.tv/cb/achievements/validator/model/query"
	"code.justin.tv/cb/achievements/validator/task/logger"
	"code.justin.tv/cb/achievements/validator/task/secrets"
	"code.justin.tv/cb/achievements/validator/task/stats"
	"code.justin.tv/cb/achievements/validator/task/validator"
	_ "github.com/lib/pq"
)

func main() {
	config.Load()

	queryName := os.Getenv("QUERY_NAME")
	loggerClient := logger.NewClient(query.Name(queryName))

	sandstormManager, err := sandstorm.NewManager(config.Values.Sandstorm.RoleARN, config.Values.AWSRegion)
	if err != nil {
		loggerClient.Panicf("failed to create sandstorm manager", err)
	}

	dirtyTahoePassword, err := sandstormManager.GetSecret(config.Values.Sandstorm.SecretNames.DirtyTahoeReplicaPassword)
	if err != nil {
		loggerClient.Panicf("failed to retrieve tahoe password from sandstorm", err)
	}

	// Noop stats client
	statsClient := stats.NewClient()

	dirtyClient, err := redshift.NewClient(&redshift.Config{
		Host:     config.Values.DirtyTahoeReplica.Host,
		User:     config.Values.DirtyTahoeReplica.User,
		Password: dirtyTahoePassword,
		Port:     config.Values.DirtyTahoeReplica.Port,
		DBName:   config.Values.DirtyTahoeReplica.DBName,
	}, statsClient)
	if err != nil {
		loggerClient.Panicf("unable to create dirty redshift client", err)
	}

	secretsClient, err := secrets.NewClient(config.Environment, config.Values.AWSRegion)
	if err != nil {
		loggerClient.Panicf("failed to create secretsmanager client", err)
	}

	cleanSecret, err := secretsClient.GetTahoeReplicaSecret(context.Background(), config.Values.TahoeReplica.SecretID)
	if err != nil {
		loggerClient.Panicf("failed to get clean secret", err)
	}

	cleanClient, err := redshift.NewClient(&redshift.Config{
		Host:     cleanSecret.Host,
		User:     cleanSecret.Username,
		Password: cleanSecret.Password,
		Port:     strconv.Itoa(cleanSecret.Port),
		DBName:   cleanSecret.DBName,
	}, statsClient)
	if err != nil {
		loggerClient.Panicf("unable to create clean redshift client", err)
	}

	validatorClient, err := validator.NewClient(validator.Params{
		CleanClient: cleanClient,
		DirtyClient: dirtyClient,
		Logger:      loggerClient,
	})
	if err != nil {
		loggerClient.Panicf("unable to create validator client", err)
	}

	switch query.Name(queryName) {
	case query.BroadcastNHours30Days:
		err := validatorClient.BroadcastNHours30Days(context.Background())
		if err != nil {
			loggerClient.Panicf("validator failed", err)
		}
	case query.NBroadcastDays30Days:
	case query.NConcurrents30Days:
	case query.NFollowers:
	case query.NRaidConsecutiveBroadcast:
	case query.NUniqueChatterBroadcast:
	}

	loggerClient.Print("success!")
}
